/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.encryptor.pgp.guest;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.json.JSONException;
import org.json.JSONObject;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.guest.GuardGuestService;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.mime.encryptor.Encryptor;
import com.openexchange.guard.mime.encryptor.exceptions.MimeEncryptorExceptionCodes;
import com.openexchange.guard.mime.encryptor.osgi.Services;
import com.openexchange.guard.mime.helpfiles.HelpFileService;
import com.openexchange.guard.mime.services.GuardParsedMimeMessage;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.oxapi.sharing.EmailShareLink;
import com.openexchange.guard.pgpcore.services.PGPCryptoService;
import com.openexchange.guard.user.OXGuardUser;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.pgp.keys.tools.PGPKeysUtil;
import com.openexchange.pgp.mail.PGPMimeService;

/**
 * {@link PGPGuestEncryptor}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class PGPGuestEncryptor extends Encryptor {

    private PGPMimeService pgpMimeService;
    private HelpFileService helpFileService;
    private GuardGuestService guestService;

    /**
     * Initializes a new {@link PGPGuestEncryptor}.
     * @param pgpCryptoService The {@link PGPCryptoService} to use for encryption
     * @param msg
     * @param recips
     * @param userid
     * @param cid
     * @param password
     */
    public PGPGuestEncryptor(PGPCryptoService pgpCryptoService,
                             PGPMimeService pgpMimeService,
                             GuardGuestService guestService,
                             HelpFileService helpFileService,
                             GuardParsedMimeMessage msg,
                             ArrayList<RecipKey> recips,
                             int userid,
                             int cid,
                             String password) {
        super(pgpCryptoService, msg, recips, userid, cid, password);
        this.pgpMimeService = Objects.requireNonNull(pgpMimeService, "pgpMimeService must not be null");
        this.guestService = guestService;
        this.helpFileService = helpFileService;

    }

    @Override
    public MimeMessage doEncrypt () throws OXException {
        return null;
    }

    private GuardKeys getSigningKey() throws OXException {
        GuardKeyService gk = Services.getService(GuardKeyService.class);
        return gk.getKeys(userid, cid);
    }

    private List<PGPPublicKey> toPGPPublicKeys(List<RecipKey> recipients){
        //Getting recipient keys suitable for encryption
        List<PGPPublicKey> pgpRecipientKeysList = new ArrayList<PGPPublicKey>();
        for (RecipKey recipientKey : recipients) {
            PGPPublicKey encryptionKey = recipientKey.getEncryptionKey();
            if(encryptionKey != null) {
                pgpRecipientKeysList.add(encryptionKey);
            }
        }
        return pgpRecipientKeysList;
    }

    private byte[] encrypt() throws OXException, JSONException, IOException, MessagingException {
        MimeMessage message = msg.getMessage();
        MimeMessage encryptedMessage = null;
        if(msg.isSign()) {
            GuardKeys signingKey = getSigningKey();
            if (signingKey == null) {
                throw MimeEncryptorExceptionCodes.SEND_UNABLE_FIND_KEYS.create("Unable to find signing keys");
            }
            PGPSecretKey pgpSigningKey = PGPKeysUtil.getSigningKey(signingKey.getPGPSecretKeyRing());
            char[] hashedPassword = CipherUtil.getSHA(password, signingKey.getSalt()).toCharArray();
            encryptedMessage = this.pgpMimeService.encryptSigned(message, pgpSigningKey, hashedPassword, toPGPPublicKeys(this.recips), helpFileService.getHelpFiles(recips));
        }
        else {
            encryptedMessage = this.pgpMimeService.encrypt(message, toPGPPublicKeys(this.recips), helpFileService.getHelpFiles(recips));
        }


        ByteArrayOutputStream ret = new ByteArrayOutputStream();
        encryptedMessage.writeTo(ret);
        return ret.toByteArray();
    }

    public void sendInvitationByMail(RecipKey recipient, GuardParsedMimeMessage msg, int templateId) throws OXException {

        final String fromEmail = msg.getFromAddress().getAddress();
        final String fromName = msg.getSenderName();
        final String guestMessage = msg.getGuestMessage();
        final String host = null;

        //Create the Guest invitation email
        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        JsonObject inviteEmail = mailCreatorService.getPasswordEmail(
            recipient.getEmail(),
            fromName,
            fromName,
            fromEmail,
            recipient.getNewGuestPass(),
            recipient.getLang(),
            templateId,
            guestMessage,
            host,
            userid,
            cid);

        //Send the invitation mail
        GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
        guardNotificationService.send(inviteEmail, msg.getSenderIP());
    }

    /**
     * Initialises an OX Guard Guest account
     *
     * @throws OXException
     */
    private void inviteGuests(int templateId) throws OXException {
        GuardConfigurationService guardConfigurationService = Services.getService(GuardConfigurationService.class);
        if (guardConfigurationService.getBooleanProperty(GuardProperty.newGuestsRequirePassword)) {
            for (final RecipKey recipient : recips) {
                //Check for new guests
                if (recipient.isGuest() && recipient.getNewGuestPass() != null && !recipient.getNewGuestPass().isEmpty()) {
                    // Mark the key that password prompt will be required.  We just need to have question non-null
                    GuardKeyService keyStorage = Services.getService(GuardKeyService.class);
                    GuardKeys guestKey = keyStorage.getKeys(recipient.getUserid(), recipient.getCid());
                    if (guestKey != null) {
                        guestKey.setQuestion("");
                        keyStorage.updateAnswerQuestionForUser(guestKey);
                    }
                    sendInvitationByMail(recipient, msg, templateId);
                }
            }
        }
    }

    private String createShareLink(String senderSessionId, String emailId, RecipKey recipient) throws OXException {
        return new EmailShareLink().createEmailShare(senderSessionId, recipient.getEmail(), emailId, recipient.getLang());
    }

    @Override
    public boolean doEncryptAndSend () throws OXException {
        //Invite new guests before sending the original message
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        int templId = guardConfigService.getIntProperty(GuardProperty.templateID, userid, cid);
        inviteGuests(templId);

        byte[] encrypted;
        try {
            encrypted = encrypt();
        } catch (JSONException e) {
            throw GuardCoreExceptionCodes.JSON_ERROR.create(e, e.getMessage());
        } catch (IOException e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        } catch (MessagingException e) {
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }

        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        String host = msg.getHost();
        String messageId = getMessageID();
        ArrayList<BodyPart> bpAttachments = new ArrayList <BodyPart>();
        final BodyPart att = new MimeBodyPart();

        try {
            att.setText(new String(encrypted, "UTF-8"));
            att.setHeader("Content-Type", "mail/rfc822");
            att.setDisposition("attachment");
            att.setFileName("pgpMail.eml");
        } catch (MessagingException | UnsupportedEncodingException e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        }
        bpAttachments.add(att);

        for (RecipKey recip : recips) {
            String id = createId();
            id = "pgp-" + id;  // After storing, add the pgp identifier

            String guestMessage = recip.isNewCreated() ? msg.getGuestMessage() : "";  // Only attach Guest message for new users
            JsonObject email = mailCreatorService.createBlankGuestMail(recip.getLang(), templId, messageId, host, userid, cid, guestMessage);
            if (email == null) {
                // TODO THROW ERROR
                return false;
            }
            email = mailCreatorService.addTo(email, recip.getName(), recip.getEmail(), recip.getType());
            email = mailCreatorService.addFrom(email, msg.getFromAddress().getPersonal(), msg.getFromAddress().getAddress());
            email = mailCreatorService.addSubject(email, msg.getSubject());
            if (recip.isNewCreated()) {
                email = mailCreatorService.addPlainText(email, guestMessage);
            }
            email = mailCreatorService.noSave(email);

            email = mailCreatorService.addURL(email, createShareLink(msg.getSenderSessionId(), id, recip));

            storeEmail(id, recip, encrypted);

            GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
            guardNotificationService.send(email, bpAttachments, "", cid, recip.getLang(), messageId, msg.getSenderIP());
        }
        return true;
    }

    private String createId() {
        String uid = CipherUtil.getUUID();
        return uid;
    }

    private void storeEmail (String id, RecipKey recip, byte[] encrypted) throws OXException {
        final UserIdentity guestUser = new UserIdentity(recip.getEmail()).setOXGuardUser(new OXGuardUser(recip.getUserid(), recip.getCid()));
        this.guestService.appendMessage(guestUser, new ByteArrayInputStream(encrypted), id, GuardGuestService.DEFAULT_FOLDER);
    }

    /**
     * Get the message identifier
     *
     * @return the message identifier
     * @throws OXException if the GuardConfigurationService is absent
     */
    protected String getMessageID() throws OXException {
        final GuardConfigurationService guardConfigurationService = Services.getService(GuardConfigurationService.class);
        String mailIdDomain = guardConfigurationService.getProperty(GuardProperty.mailIdDomain);
        final String oxURL = guardConfigurationService.getProperty(GuardProperty.externalEmailURL);
        if (mailIdDomain.equals("")) {  // If maildomain not defined, try to extract domain from URL
            if (oxURL.contains("/")) {
                mailIdDomain = oxURL.substring(0, oxURL.indexOf("/"));
            }
            if (mailIdDomain.contains(":")) {
                mailIdDomain = mailIdDomain.substring(0, mailIdDomain.indexOf(":"));
            }
        }
        if (mailIdDomain.equals("")) {
            mailIdDomain = "GUARD";
        }  // Otherwise, just define it as Guard
        return ("<" + CipherUtil.getUUID() + "@" + mailIdDomain + ">");
    }

}
