/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.milter.actions.encryptor;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import org.bouncycastle.openpgp.PGPPublicKey;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.milter.service.MilterMail;
import com.openexchange.guard.pgp.GuardPGPMimeMailCreator;
import com.openexchange.pgp.core.PGPEncrypter;

/**
 * {@link DoEncrypt}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class DoEncrypt {

    private String contentType = null;

    public byte[] encrypt (MilterMail mail) throws Exception {
        // Write the data as Mime Email
        MimeMessage message = getMessage(mail.getBodyBytes());
        ByteArrayOutputStream out = new ByteArrayOutputStream();

        // Add mime headers back
        addHeaders (mail, out);

        message.writeTo(out);
        InputStream in = new ByteArrayInputStream(out.toByteArray());
        ByteArrayOutputStream encrypted = new ByteArrayOutputStream();
        // List of recipients to array
        List<PGPPublicKey> pgpRecipientKeysList = new ArrayList<PGPPublicKey>();
        for (RecipKey recipientKey : mail.getKeys()) {
            pgpRecipientKeysList.add(recipientKey.getEncryptionKey());
        }
        PGPPublicKey[] pgpRecipientKeys = pgpRecipientKeysList.toArray(new PGPPublicKey[pgpRecipientKeysList.size()]);
        // Encrypt it
        new PGPEncrypter().encrypt(in, encrypted, true, pgpRecipientKeys);
        // Wrap it in new Mime Body
        MimeMessage wrapped = createWrapper(encrypted.toByteArray());
        contentType = wrapped.getContentType();
        ByteArrayOutputStream wrapped_out = new ByteArrayOutputStream();
        // Print it without headers to output
        wrapped.writeTo(wrapped_out, new String[] {"Content-Type", "MIME-Version", "Message-ID"});
        byte[] wrapped_bytes = wrapped_out.toByteArray();

        return wrapped_bytes;
    }

    private void checkAndAddHeader (String header, MilterMail mail, ByteArrayOutputStream out) throws IOException {
        if (mail.getHeader(header) != null) {
            String headerString = header + ": " + mail.getHeader(header) + "\r";
            out.write(headerString.getBytes(StandardCharsets.UTF_8));
        }

    }
    private void addHeaders (MilterMail mail, ByteArrayOutputStream out) throws IOException {
        checkAndAddHeader ("Message-ID", mail, out);
        checkAndAddHeader ("MIME-Version", mail, out);
        checkAndAddHeader ("Content-Type", mail, out);
        String space = "\r\r";
        out.write(space.getBytes(StandardCharsets.UTF_8));

    }

    private MimeMessage getMessage (byte[] data) throws MessagingException {
        MimeMessage message = new MimeMessage(Session.getDefaultInstance(new Properties()), new ByteArrayInputStream(data));
        return message;
    }

    private MimeMessage createWrapper (byte[] data) throws Exception {
        GuardPGPMimeMailCreator mailCreator = new GuardPGPMimeMailCreator();
        MimeMultipart mp = mailCreator.createPGPMime(data, null);
        MimeMessage new_message = new MimeMessage(Session.getDefaultInstance(new Properties()));
        new_message.setContent(mp);
        new_message.saveChanges();
        return new_message;
    }

    public String getContentType() {
        return this.contentType;
    }
}
