/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services.impl.lookup;

import java.util.Collection;
import java.util.Iterator;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.hkpclient.services.HKPClientService;
import com.openexchange.guard.hkpclient.services.RemoteKeyResult;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.services.impl.osgi.Services;
import com.openexchange.guard.keymanagement.services.lookup.AbstractRecipKeyLookupStrategy;
import com.openexchange.guard.keymanagement.services.lookup.RecipKeyLookupStrategy;
import com.openexchange.pgp.keys.tools.PGPKeysUtil;

/**
 * {@link HKPRecipKeyLookupStrategy} performs a HKP lookup in order to find a {@link RecipKey}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class HKPRecipKeyLookupStrategy extends AbstractRecipKeyLookupStrategy {

    private final String hkpClientToken;

    /**
     * Initializes a new {@link HKPRecipKeyLookupStrategy}.
     * @param hkpClientToken The client token to use for outgoing HKP requests
     */
    public HKPRecipKeyLookupStrategy(String hkpClientToken) {
        this.hkpClientToken = hkpClientToken;
    }

    /**
     * Internal method to convert a HKP {@link RemoteKeyResult} into a {@link RecipKey}.
     *
     * @param email The email
     * @param remoteResult The results to convert to a {@link RecipKey}
     * @return The {@link RecipKey} constructed of the given {@link RemoteKeyResult}
     */
    private RecipKey toRecipKey(String email, RemoteKeyResult remoteResult) {
        PGPPublicKeyRing publicKeyRing = remoteResult.getRing();
        RecipKey recipKey = new RecipKey(remoteResult.getSource());
        recipKey.setPGPPrivateKeyRing(publicKeyRing);
        recipKey.setEmail(email);
        recipKey.setPgp(true);
        recipKey.setExpired(PGPKeysUtil.checkAllExpired(publicKeyRing));
        return recipKey;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.keymanagement.services.impl.RecipKeyLookupStrategy#lookup(java.lang.String)
     */
    @Override
    public RecipKey lookup(String email, int timeout) throws OXException {
        Collection<RemoteKeyResult> results = Services.getService(HKPClientService.class).find(hkpClientToken, email, timeout);
        if(results != null && !results.isEmpty()) {
            Iterator<RemoteKeyResult> it = results.iterator();
            while (it.hasNext()) {
                RemoteKeyResult result = it.next();
                if (result != null && hasValidEncryptionKey(result.getRing())) {
                    return toRecipKey(email, result);
                }
            }
        }
        return null;
    }
}
