/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import org.bouncycastle.openpgp.PGPPublicKey;
import com.openexchange.contact.ContactService;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.container.Contact;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.OGPGPKeyRing;
import com.openexchange.guard.keymanagement.commons.export.KeyExportUtil;
import com.openexchange.guard.keymanagement.services.ContactKeyService;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.PublicExternalKeyService;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.pgp.keys.parsing.PGPPublicKeyRingFactory;
import com.openexchange.pgp.keys.tools.PGPKeysUtil;
import com.openexchange.session.Session;

/**
 * {@link ContactKeyServiceImpl}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class ContactKeyServiceImpl implements ContactKeyService {

    private final GuardKeyService          keyService;
    private final PublicExternalKeyService externalKeyService;
    private final ContactService           contactService;
    private final GuardRatifierService     ratifierService;

    /**
     * Initializes a new {@link ContactKeyServiceImpl}.
     *
     * @param keyService The {@link GuardKeyService} to use
     * @param externalKeyService The {@link PublicExternalKeyService} to use
     * @param contactService The {@link ContactService} to use
     */
    public ContactKeyServiceImpl(GuardKeyService keyService, PublicExternalKeyService externalKeyService, ContactService contactService, GuardRatifierService ratifierService) {
        this.keyService = Objects.requireNonNull(keyService, "keyService must not be null");
        this.externalKeyService = Objects.requireNonNull(externalKeyService, "externalKeyService must not be null");
        this.contactService = Objects.requireNonNull(contactService, "contactService must not be null");
        this.ratifierService = Objects.requireNonNull(ratifierService, "Ratifier service must not be null");
    }

    /**
     * Internal factory method to create a session for the given user
     *
     * @param userId The user id
     * @param contextId the context id
     * @return A "fake" session based on the given parameters.
     */
    private Session createaFakeSession(int userId, int contextId) {

        return new Session() {

            private final ConcurrentMap<String, Object> parameters = new ConcurrentHashMap<String, Object>(8, 0.9f, 1);

            @Override
            public void setParameter(String name, Object value) {
                if (null == value) {
                    parameters.remove(name);
                } else {
                    parameters.put(name, value);
                }
            }

            @Override
            public void setLocalIp(String ip) {
                // NO-OP
            }

            @Override
            public void setHash(String hash) {
                // NO-OP
            }

            @Override
            public void setClient(String client) {
                // NO-OP
            }

            @Override
            public boolean isTransient() {
                return false;
            }

            @Override
            public String getUserlogin() {
                return null;
            }

            @Override
            public int getUserId() {
                return userId;
            }

            @Override
            public String getSessionID() {
                return null;
            }

            @Override
            public String getSecret() {
                return null;
            }

            @Override
            public String getRandomToken() {
                return null;
            }

            @Override
            public String getPassword() {
                return null;
            }

            @Override
            public Set<String> getParameterNames() {
                return parameters.keySet();
            }

            @Override
            public Object getParameter(String name) {
                return parameters.get(name);
            }

            @Override
            public String getLoginName() {
                return null;
            }

            @Override
            public String getLogin() {
                return null;
            }

            @Override
            public String getLocalIp() {
                return null;
            }

            @Override
            public String getHash() {
                return null;
            }

            @Override
            public int getContextId() {
                return contextId;
            }

            @Override
            public String getClient() {
                return null;
            }

            @Override
            public String getAuthId() {
                return null;
            }

            @Override
            public boolean containsParameter(String name) {
                return parameters.containsKey(name);
            }
        };
    }

    private String getKeyIdsString(GuardKeys key) {
        Iterator<PGPPublicKey> pubKeys = key.getPGPPublicKeyRing().getPublicKeys();
        StringBuilder sb = new StringBuilder();
        while (pubKeys.hasNext()) {
            PGPPublicKey pkey = pubKeys.next();
            String id = PGPKeysUtil.getFingerPrint(pkey.getFingerprint());
            sb.append(id.substring(id.length() - 8));
            if (pubKeys.hasNext()) {
                sb.append(" ");
            }
        }
        return sb.toString();
    }

    /**
     * Validate if each of the three contact emails are valid.
     * If so, adds to list and returns (without duplicates)
     * @param contact
     * @return
     */
    private List<String> getContactEmails(Contact contact) {
        ArrayList<String> emails = new ArrayList<String>();
        if (ratifierService.isValid(contact.getEmail1())) {
            emails.add(contact.getEmail1());
        }
        if (ratifierService.isValid(contact.getEmail2())) {
            emails.add(contact.getEmail2());
        }
        if (ratifierService.isValid(contact.getEmail3())) {
            emails.add(contact.getEmail3());
        }
        return emails.stream().distinct().collect(Collectors.toList());
    }
    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keymanagement.services.ContactKeyService#getKeysForContact(int, int, java.lang.String, java.lang.String)
     */
    @Override
    public List<OGPGPKeyRing> getKeysForContact(int userId, int contextId, String contact_id, String folder_id) throws OXException {
        Contact contact = contactService.getContact(createaFakeSession(userId, contextId), folder_id, contact_id);
        List<String> emails = getContactEmails(contact);
        final boolean isGuardKey = true;
        List<OGPGPKeyRing> ret = emails.isEmpty() ? Collections.emptyList() : this.externalKeyService.get(userId, contextId, emails);

        try {
            for (String email : emails) {
                GuardKeys key = keyService.getKeys(email);
                if (key != null) {
                    ret.add(new OGPGPKeyRing(getKeyIdsString(key), PGPPublicKeyRingFactory.create(KeyExportUtil.export(key.getPGPPublicKeyRing())),
                        isGuardKey));
                }
            }
        } catch (IOException e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }

        return ret;
    }
}
