/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.support;

import java.net.URI;
import com.openexchange.exception.OXException;
import com.openexchange.guard.support.PasswordReseter.EmailTarget;

/**
 * Interface for OX Guard Support functionality
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public interface SupportService {

    /**
     * Marks a deleted key as "exposed", which allows a user to to download a key
     * @param contextId the context ID of the user
     * @param email the email of the user
     * @return an URI to access the deleted key for downloading it
     * @throws OXException
     */
    public URI exposeKey(int contextId, String email) throws OXException;

    /**
     * Resets a user's password, a new password will be sent to the user's primary or secondary email address
     * @param contextId the context ID
     * @param email the email
     * @param language to use for creating the reset email
     * @throws OXException
     */
    public EmailTarget resetPassword(String email, String language) throws OXException;

    /**
     * Deletes a user and all of his keys. Deleted keys are backed up and can still be accessed as download by using {@link #ExposeKey}
     * @param contextId the context id
     * @param userId the user's id
     * @throws OXException
     */
    public void deleteUser(int contextId, int userId) throws OXException;

    /**
     * Deletes a user and all of his keys. Deleted keys are backed up and can still be accessed as download by using {@link #ExposeKey}
     * @param email the email address of the user which should be deleted
     * @throws OXException
     */
    public void deleteUser(String email) throws OXException;

    /**
     * Removes the secret PIN from a guest key
     * @param email the email address of the guest to remove the pin for
     * @throws OXException
     */
    public void removePin(String email) throws OXException;

    /**
     * Upgrade a guest account to a regular user
     * @param email the email addres sof the guest which should be upgraded to a regular user
     * @throws OXException
     */
    public void upgradeGuestAccount(String email, String userid, String cid) throws OXException;
}
