/*
 *
 *    OPEN-XCHANGE legal inforsemation
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgpmail;

import java.io.IOException;
import java.io.InputStream;
import javax.mail.MessagingException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItemIterator;
import org.apache.commons.fileupload.FileItemStream;
import org.apache.commons.fileupload.FileUploadException;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.inputvalidation.RangeInputValidator;
import com.openexchange.guard.mime.services.GuardParsedMimeMessage;
import com.openexchange.guard.mime.services.MimeEncryptionService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.fileupload.FileUploadHandler;
import com.openexchange.guard.servlets.fileupload.JsonFileUploadHandler;
import com.openexchange.guard.user.UserIdentity;

/**
 * Encrypt an email
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.4.0
 */
public class MimeEmailAction extends GuardServletAction {

	private static final String JSON_DATA_FIELD_NAME = "json";
	private static final String JSON_AUTH_FIELD_NAME = "guardAuth";
	private static final String MESSAGE_FIELD_NAME = "file";

    private GuardParsedMimeMessage parseMimeMessage(HttpServletRequest request, JSONObject json, InputStream message, int userId, int contextId, String fromName, String fromEmail, boolean isGuest) throws MessagingException, JSONException, OXException, FileUploadException, IOException {
        return ParseIncoming.parse(request, json, message, userId, contextId, fromName, fromEmail, isGuest);
    }

    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        RangeInputValidator<Integer> positiveValidator = new RangeInputValidator<Integer>(0, Integer.MAX_VALUE);

        //We trust the userId and contextId, because it comes from a backend callback
        final boolean mandatory = true;
        final int userId = positiveValidator.assertInput(ServletUtils.getIntParameter(request, "user", mandatory), "user");
        final int contextId = positiveValidator.assertInput(ServletUtils.getIntParameter(request, "context", mandatory), "context");
        final String fromName = ServletUtils.getStringParameter(request, "fromName", mandatory);
        final String senderEmail = ServletUtils.getStringParameter(request, "email", mandatory);
        final boolean senderIsGuest =  ServletUtils.getBooleanParameter(request, "isGuest");

        JSONObject json = null;
        JSONObject guardAuthJson = null;
        InputStream messageStream = null;

        //Using streaming approach because we do not want plaintext data to be buffered on disk for parsing.
        FileUploadHandler fileUploadHandler = new FileUploadHandler(Services.getService(GuardConfigurationService.class));
        JsonFileUploadHandler jsonFileUploadHandler = new JsonFileUploadHandler(fileUploadHandler);
        FileItemIterator itemIterator = fileUploadHandler.parseIterator(request);
        while(itemIterator.hasNext()){
        	FileItemStream fileItemStream = itemIterator.next();
        	if(fileItemStream.getFieldName().equals(JSON_DATA_FIELD_NAME)){
        		try(InputStream jsonStream = fileItemStream.openStream()){
        			json = jsonFileUploadHandler.getJsonFrom(jsonStream);
        		}
        	}
        	else if(fileItemStream.getFieldName().equals(JSON_AUTH_FIELD_NAME)){
        		try(InputStream jsonStream = fileItemStream.openStream()){
        			guardAuthJson = jsonFileUploadHandler.getJsonFrom(jsonStream);
        		}
        	}
        	else if(fileItemStream.getFieldName().equals(MESSAGE_FIELD_NAME)){
        		messageStream = fileItemStream.openStream();
        	}

        	if(json != null && messageStream != null){
        		//we got all we need: prevent that hasNext() skips the dataStream.
        		//note: guardAuthJson might be missing if the mail should not get signed
        		break;
        	}
        }

		UserIdentity signingIdentity = guardAuthJson != null
				? getUserIdentityFrom(guardAuthJson)
				: null;
        if (guardAuthJson != null && signingIdentity == null) {
            throw GuardAuthExceptionCodes.BAD_PASSWORD.create();
        }

        //Extracting user identity from the authenticationToken
        String password = signingIdentity != null ? new String(signingIdentity.getPassword()) : null;

        final GuardParsedMimeMessage parsedMimeMessage = parseMimeMessage(request, json, messageStream, userId, contextId, fromName, senderEmail, senderIsGuest);
        MimeEncryptionService mimeEncryptor = Services.getService(MimeEncryptionService.class);
        mimeEncryptor.doEncryption(parsedMimeMessage, userId, contextId, password, response.getOutputStream());
    }
}
