/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.keymanagement.responses;

import java.util.ArrayList;
import java.util.Iterator;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keymanagement.commons.export.KeyExportUtil;
import com.openexchange.pgp.keys.tools.PGPKeysUtil;

/**
 * {@link PublicKeyRingInfoResponse}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class PublicKeyRingInfoResponse {

    private final PGPPublicKeyRing keyRing;
    private final PGPSecretKeyRing pgpSecretKeyRing;

    /**
     * Initializes a new {@link PublicKeyRingInfoResponse}.
     *
     * @param pgpring
     */
    public PublicKeyRingInfoResponse(PGPPublicKeyRing keyRing) {
        this(keyRing, null);
    }

    /**
     * Initializes a new {@link PublicKeyRingInfoResponse}.
     *
     * @param keyRing
     * @param pgpSecretKeyRing
     */
    public PublicKeyRingInfoResponse(PGPPublicKeyRing keyRing, PGPSecretKeyRing pgpSecretKeyRing) {
        this.keyRing = keyRing;
        this.pgpSecretKeyRing = pgpSecretKeyRing;
    }

    public String getRing() throws OXException {
        return KeyExportUtil.export(this.keyRing);
    }

    /**
     * Checks if the given public key has a corresponding, not empty, private key
     *
     * @param publicKey The public key to check for a corresponding private key.
     * @return true, if the public key has a corresponding, not empty, private key, false otherwise.
     */
    private boolean checkHasPrivate(PGPPublicKey publicKey) {
        if (pgpSecretKeyRing != null) {
            PGPSecretKey secretKey = pgpSecretKeyRing.getSecretKey(publicKey.getKeyID());
            return secretKey != null && !secretKey.isPrivateKeyEmpty();
        }
        return false;
    }

    public PublicKeyInfoResponse[] getKeys() {
        ArrayList<PublicKeyInfoResponse> list = new ArrayList<PublicKeyInfoResponse>();
        Iterator<PGPPublicKey> iter = this.keyRing.getPublicKeys();
        PGPPublicKey keyUsedForEncryption = PGPKeysUtil.getEncryptionKey(keyRing);
        while (iter.hasNext()) {
            PGPPublicKey publicKey = iter.next();
            final boolean hasPrivateKey = checkHasPrivate(publicKey);
            final boolean usedForEncryption = keyUsedForEncryption != null &&
                keyUsedForEncryption.getKeyID() == publicKey.getKeyID();
            list.add(new PublicKeyInfoResponse(publicKey,hasPrivateKey, usedForEncryption));
        }
        return list.toArray(new PublicKeyInfoResponse[list.size()]);
    }
}
