/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.guest;

import java.io.InputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.google.gson.JsonObject;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.inputvalidation.UUIDValidator;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.PasswordChangeService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.guest.exceptions.GuestServletExceptionCodes;
import com.openexchange.guard.storage.Storage;

/**
 * {@link GuestFirstPasswordAction}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v7.8.3
 */
public class GuestFirstPasswordAction extends GuardServletAction {

    /**
     * For a guest users first access, change temporary password saved in recovery
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        GuardConfigurationService guardConfig = Services.getService(GuardConfigurationService.class);
        if (guardConfig.getBooleanProperty(GuardProperty.newGuestsRequirePassword)) {  // If first password required, then we don't use this service
            throw GuestServletExceptionCodes.SERVICE_UNAVAILABLE.create();
        }
        GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
        String ip = ServletUtils.getClientIP(request);
        if (antiabuse.isBad(ip)) {
            throw GuestServletExceptionCodes.LOCKOUT.create();
        }
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        String newpass = null;
        String question = null;
        String answer = null;
        String email = null;
        String itemId = null;
        try {
            newpass = json.get("newpass").getAsString();
            question = json.get("question").getAsString();
            answer = json.get("answer").getAsString();
            email = new EmailValidator().assertInput(json.get("email").getAsString(), "email");
            itemId = new UUIDValidator("pgp-").assertInput(json.get("itemId").getAsString(),"itemId");
        } catch (NullPointerException e) {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create("JSON requires newpass, question, answer, email, itemid");
        }
        String pin = null;
        if (json.has("pin")) {
            pin = json.get("pin").getAsString();
        }
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys keys = keyService.getKeys(email);
        if (keys == null) {
            antiabuse.addBad(ip);
            throw GuestServletExceptionCodes.NO_KEY.create(email);
        }
        if (keys.getContextid() > 0) {
            antiabuse.addBad(ip);
            throw GuestServletExceptionCodes.NOT_GUEST.create();
        }
        if (keys.getQuestion() != null && keys.getQuestion().equals("PIN")) {  // If PIN provided, check that it is valid
            antiabuse.addBad(ip);
            if (pin == null || !keys.getAnswer().equals(pin.trim())) {
                throw GuestServletExceptionCodes.PIN_REQUIRED.create();
            }
        }

        // Here, let's verify the itemId exists
        Storage storage = Services.getService(Storage.class);
        try(InputStream in = storage.readObj(itemId);){
            if (in == null) {
                antiabuse.addBad(ip);
                throw GuestServletExceptionCodes.BAD_EMAIL_ID.create();
            }
        }

        // OK, verified.  Let's change the password
        PasswordChangeService passService = Services.getService(PasswordChangeService.class);
        OxCookie cookie = new OxCookie(request.getCookies());
        String retJson = passService.changePassword(keys.getUserid(), keys.getContextid(), email, "", newpass, question, answer, pin, cookie.getOxReaderID());
        ServletUtils.sendOK(response, "application/json", retJson);



    }

}
