/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.guest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.GuardMasterKeyService;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.OxCookie;


public class Guest {

    private static Logger LOG = LoggerFactory.getLogger(Guest.class);

    /**
     * Get password reset question
     *
     * @param request
     * @param response
     * @param cookie
     * @throws IOException
     */
    public void getQuestion(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws OXException {
        String email = new EmailValidator().assertInput(ServletUtils.getStringParameter(request, "email", true), "email");
        EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);

        //Getting the mail mapping
        String question = "nf";
        Email emailKeyMapping = ogEmailStorage.getByEmail(email);
        if (emailKeyMapping != null) {
            //Getting the guest's key
            KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
            GuardKeys key = ogKeyTableStorage.getKeyForEmailAndContext(email, emailKeyMapping.getContextId());
            if (key != null) {
                question = key.getQuestion();
                if (!key.isRecoveryAvail()) {
                    question = "nr";
                }
            }
        } else {
            LOG.error(String.format("No email mapping found for email %s", email));
        }
        ServletUtils.sendHtmlOK(response, question);
    }

    /**
     * Perform password reset after question answered
     *
     * @param request
     * @param response
     * @param cookie
     * @throws IOException
     * @throws OXException
     */
    public void resetPassword(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws OXException {
        String email = new EmailValidator().assertInput(ServletUtils.getStringParameter(request, "email", true), "email");
        String answer = ServletUtils.getStringParameter(request, "answer", true);
        String lang = ServletUtils.getStringParameter(request, "language");
        Integer templId = ServletUtils.getIntParameter(request, "templid");
        if(templId == null) {
            templId = 0;
        }

        String question = "";

        int cid = 0;
        int userid = 0;
        try {
            GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
            GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
            if (antiabuse.isBad("reset-" + email, configService.getIntProperty(GuardProperty.badPasswordCount))) {
                ServletUtils.sendHtmlOK(response, "lockout");
                return;
            }

            KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
            GuardShardingService shardingService = Services.getService(GuardShardingService.class);
            GuardKeys key = ogKeyTableStorage.getKeyForGuest(email, shardingService.getShard(email));

            String prevanswer = null;
            if (key != null) {
                prevanswer = key.getAnswer();
                userid = key.getUserid();
                cid = key.getContextid();
                question = key.getQuestion();
            }
            if (question != null) {  // PIN is not used for answer
                if (question.equals("PIN")) {
                    prevanswer = null;
                }
            }
            if (prevanswer != null) {
                GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
                if (Strings.isEmpty(question)) {
                    question = "default";
                }
                String rc = Services.getService(GuardMasterKeyService.class).getMasterKey().getRC();
                prevanswer = cipherService.decrypt(prevanswer, rc, question);
                if (prevanswer == null) {
                    ServletUtils.sendNotAcceptable(response, "Error decoding answer");
                    return;
                }
                prevanswer = prevanswer.toLowerCase().replace(" ", "");
                answer = answer.toLowerCase().replace(" ", "");
                if (!prevanswer.equals(answer)) {
                    ServletUtils.sendHtmlOK(response, "No Match");
                    antiabuse.addBad("reset-" + email);
                    return;
                }
            }
            String senderIp = request.getRemoteAddr();
            reset(email, lang, templId, cid, userid, request.getServerName(), senderIp);

            antiabuse.removeBad(email);
            ServletUtils.sendHtmlOK(response, "OK");
        } catch (Exception ex) {
            LOG.error("Error resetting password for email account " + email, ex);
            ServletUtils.sendNotAcceptable(response, "Failed to change");
        }
    }

    protected void reset(String email, String lang, int templId, int cid, int userid, String host, String senderIp) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);

        boolean recoveryEnabled = keyService.isRecoveryEnabled(userid, cid);
        if (!recoveryEnabled) {  // No recovery available
            throw GuardCoreExceptionCodes.DISABLED_ERROR.create("reset password");
        }

        String newpass = keyService.generatePassword();

        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        JsonObject mail = mailCreatorService.getResetEmail(email, mailCreatorService.getFromAddress(email, email, userid, cid), newpass, lang, templId, host, userid, cid);

        GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
        guardNotificationService.send(mail, senderIp);

        keyService.resetPassword(email, newpass);
    }
}
