/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.storage.cache;

import java.util.List;
import com.openexchange.exception.OXException;

/**
 * {@link FileCacheStorage} provides access to stored FileCache items
 */
public interface FileCacheStorage {

    /**
     * Gets an item by id
     *
     * @param itemId the item id
     * @return the FileCacheItem for the given itemId
     * @throws OXException due an error
     */
    FileCacheItem getById(String itemId) throws OXException;

    /**
     * Gets an item by ID for a specific user
     *
     * @param itemId The item id
     * @param userId The ID of the user to fetch the item for
     * @return The FileCacheItem, or null if there is no such item for the given user
     * @throws OXException
     */
    FileCacheItem getByIdForUser(String itemId, int userId) throws OXException;

    /**
     * Gets a list of items for a given user
     * @param userId The user ID
     * @param offset The offset from where to return the items (depends on the underlying sorting)
     * @param maxCount the maximum amount of returned items
     * @return A list of items for the given user which contains not more than <code>maxCount</code> items.
     */
    List<FileCacheItem> getForUser(int userId, int offset, int maxCount) throws OXException;

    /**
     * Creates a new FileCache item
     *
     * @param itemId the unique id to use for the new item
     * @param userId the owner of the item
     * @param the location of the item on the disk
     * @throws due an error
     */
    void insert(String itemId, int userId, String path) throws OXException;

    /**
     * Updates the last access of a FileCache item to "NOW"
     *
     * @param itemId the id of the item to update
     * @throws due an error
     */
    void updateLastDate(String itemId) throws OXException;

    /**
     * Deletes a FileItem from the cache
     *
     * @param idemId the id of the item to delete
     * @throws OXException due an error
     */
    void delete(String itemId) throws OXException;

    /**
     * Finds FileCache items which have not been accessed since x days
     *
     * @param days the amount of days
     * @return A list of items which have not been accessed since the given amount of days
     * @throws OXException due an error
     */
    List<FileCacheItem> findOld(int days) throws OXException;

    /**
     * Finds all filecache items for a given guest user
     * @param userId
     * @return
     * @throws OXException
     */
    List<FileCacheItem> findAllForUser (int userId) throws OXException;

}
