/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.services;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Properties;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimePart;
import javax.mail.internet.MimePartDataSource;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MimeParser {

    private static Logger LOG = LoggerFactory.getLogger(MimeParser.class);
    private static final String BEGIN_PGP_MESSAGE_MARKER = "-----BEGIN PGP MESSAGE-----";
    private final ArrayList<Attachment> attachments = new ArrayList<Attachment>();
    private final MimeMessage message;

    public MimeParser(final byte[] email) throws MessagingException {
        final Session s = Session.getDefaultInstance(new Properties());
        final InputStream is = new ByteArrayInputStream(email);
        message = new MimeMessage(s, is);
    }

    public MimeParser(final MimeMessage message) {
        this.message = message;
    }

    public MimeMessage getMessage() {
        return (message);
    }

    public boolean isPGPMime() throws MessagingException {
        if (message.getContentType() != null && message.getContentType().toString().toLowerCase().contains("multipart/mixed")) {
            Multipart mp;
            try {
                mp = (Multipart) message.getContent();
                final int count = mp.getCount();
                for (int i = 0; i < count; i++) {
                    final BodyPart bp = mp.getBodyPart(i);
                    if (bp.getContentType().toLowerCase().contains("pgp-encrypted")) return true;
                }
            } catch (IOException e) {
                LOG.error("Problem parsing email to check if Mime message", e);
            }
        }
        // This should not really ever return positive, as PGP MIME by standards should be multipart/mixed.  ? Return false always
        return message.getContentType() != null && message.getContentType().toString().toUpperCase().contains("APPLICATION/PGP-ENCRYPTED");
    }

    public boolean isPGPInline() throws IOException, MessagingException {

        byte[] peekedData = new byte[1024];
        int read = message.getInputStream().read(peekedData);
        if(read > 0) {
            String peekedContent = new String(peekedData, StandardCharsets.UTF_8);
            return peekedContent.contains(BEGIN_PGP_MESSAGE_MARKER);
        }

        return false;
    }

    public MimeMultipart getFirstMultipart() throws IOException, MessagingException {
        final Object content = message.getContent();
        if (content instanceof Multipart || content instanceof MimeMultipart) {
            final MimeMultipart mp = (MimeMultipart) content;
            return (mp);
        }
        return (null);
    }

    public void ParseAttachments() throws MessagingException, IOException {
        final Object content = message.getContent();
        if (content instanceof Multipart || content instanceof MimeMultipart) {
            final Multipart mp = (Multipart) content;
            parseMultipart(mp);
        }
    }

    public void parseMultipart(final Multipart mp) throws IOException, MessagingException {
        final int count = mp.getCount();
        for (int i = 0; i < count; i++) {
            final BodyPart bp = mp.getBodyPart(i);
            final Object content = bp.getContent();
            if (content instanceof String) {  // Check if this string is a text attachment, or just the message body
                if (bp.getDisposition() != null) {
                    if (bp.getDisposition().contains("attachment")) {
                        final Attachment att = new Attachment();
                        att.setContent(((String) content).getBytes("UTF-8"));
                        att.setFilename(bp.getFileName());
                        att.setType(bp.getContentType());
                        String type = att.getType();
                        if (type.contains(";")) {
                            att.setType(type.substring(0, type.indexOf(";")));
                        }
                        attachments.add(att);
                    }
                }

            } else if (content instanceof InputStream) {
                final Attachment att = new Attachment();
                att.setContent(IOUtils.toByteArray((InputStream) content));
                att.setFilename(bp.getFileName());
                att.setType(bp.getContentType());
                String type = att.getType();
                if (type.contains(";")) {
                    att.setType(type.substring(0, type.indexOf(";")));
                }
                attachments.add(att);
                // handle input stream
            } else if (content instanceof Multipart || content instanceof MimeMultipart) {
                final Multipart mp2 = (Multipart) content;
                parseMultipart(mp2);
            }
        }
    }

    public void mergeAttachments(final ArrayList<Attachment> prevattachments) {
        for (int i = 0; i < attachments.size(); i++) {
            prevattachments.add(attachments.get(i));
        }
    }

    public static String decodeMimePart(final MimePart mp) throws IOException, MessagingException {
        final ByteArrayOutputStream writer = new ByteArrayOutputStream();
        mp.writeTo(writer);
        final String section = writer.toString("UTF-8");
        final MimePartDataSource ds = new MimePartDataSource(mp);
        final ByteArrayOutputStream decoded = new ByteArrayOutputStream();
        IOUtils.copy(ds.getInputStream(), decoded);
        final int i = section.indexOf("\r\n\r\n");
        if (i > 0) {
            final StringBuilder sb = new StringBuilder();
            sb.append(section.substring(0, i));
            sb.append("\r\n\r\n");
            sb.append(decoded.toString("UTF-8"));
            return (sb.toString());
        }
        return (decoded.toString("UTF-8"));
    }

    /**
     * Get the first MIME boundry/border in a mime message
     *
     * @param msg
     * @return null if no boarder specified in content-type
     * @throws MessagingException
     */
    public static String getBoundry(final MimeMessage msg) throws MessagingException {
        try {
            final String[] content = msg.getHeader("Content-Type");
            if (content != null && content.length > 0) {
                final int i = content[0].indexOf("boundary");
                if (i > 0) {
                    String bndry = content[0].substring(i);
                    if (bndry.contains(";")) {
                        bndry = bndry.substring(0, bndry.indexOf(";"));
                    }
                    return (bndry.substring(bndry.indexOf("\"")).replace("\"", ""));
                }
            }
        } catch (final Exception e) {
            LOG.error("Error finding mime border ", e);
        }
        return (null);
    }

    /**
     * Function to parse out first section of mime body using the border
     *
     * @param mime The mime body
     * @param border The mime section border
     * @return
     */
    public static String getMimeSection(final String mime, String border) {
        if (border == null) {
            return (mime);
        }
        try {
            border = "--" + border;
            final int j = mime.indexOf(border);
            final int k = mime.indexOf(border, j + 10);
            if (j > 0 && k > 0) {
                final String data = mime.substring(j + 2 + border.length(), k - 2);
                return (data);
            }
        } catch (final Exception e) {
            LOG.error("Error parsing mime section for email");
        }

        return (mime);
    }
}
