/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.encryptor.pgp.pgpmime;

import java.io.IOException;
import java.util.List;
import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.mime.encryptor.Decryptor;
import com.openexchange.guard.mime.encryptor.exceptions.MimeEncryptorExceptionCodes;
import com.openexchange.guard.mime.encryptor.pgp.impl.HTMLStripper;
import com.openexchange.guard.pgp.signing.impl.SignatureResults;
import com.openexchange.guard.pgpcore.services.PGPMimeCryptoService;
import com.openexchange.guard.pgpcore.services.PGPMimeDecryptionResult;
import com.openexchange.guard.pgpcore.services.FromHeaderVerifier;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.pgp.core.PGPSignatureVerificationResult;

/**
 * {@link PGPMimeDecryptor}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.8.0
 */
public class PGPMimeDecryptor extends Decryptor {

    private final PGPMimeCryptoService pgpMimeCryptoService;
    private final GuardConfigurationService guardConfigService;
    private final UserIdentity         userIdentity;

    private static final org.slf4j.Logger LOG =
        org.slf4j.LoggerFactory.getLogger(PGPMimeDecryptor.class);

    /**
     * Initializes a new {@link PGPMimeDecryptor}.
     *
     * @param pgpMimeCryptoService The {@link PGPMimeCryptoService} to use for decrypting the content.
     * @param identity The identity of the user to decrypt the content for.
     */
    public PGPMimeDecryptor(PGPMimeCryptoService pgpMimeCryptoService, GuardConfigurationService guardConfigService, UserIdentity identity) {
        this.pgpMimeCryptoService = pgpMimeCryptoService;
        this.guardConfigService = guardConfigService;
        this.userIdentity = identity;
    }

    /**
     * Check for missing required headers in the encyrpted body. If missing, add from original plaintext
     *
     * @param orig
     * @param decrypted
     */
    private void checkMissingHeaders(MimeMessage orig, MimeMessage decrypted) {
        checkMissingHeader(orig, decrypted, "Date");
        checkMissingHeader(orig, decrypted, "Subject");
        checkMissingHeader(orig, decrypted, "From");
        checkMissingHeader(orig, decrypted, "To");
        checkMissingHeader(orig, decrypted, "CC");
        checkMissingHeader(orig, decrypted, "Return-Path");
    }

    /**
     * Check for missing headers in decrypted email. If not present, then add from original
     *
     * @param orig
     * @param decrypted
     */
    private void checkMissingHeader(MimeMessage orig, MimeMessage decrypted, String header) {
        try {
            if (decrypted.getHeader(header) == null) {
                if (orig.getHeader(header) != null && orig.getHeader(header).length > 0) {
                    String[] headers = orig.getHeader(header);
                    for (String data : headers) {
                        decrypted.addHeader(header, data);
                    }
                }
            }
        } catch (MessagingException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.mime.encryptor.Decryptor#doDecrypt(javax.mail.internet.MimeMessage)
     */
    @Override
    public MimeMessage doDecrypt(MimeMessage message) throws OXException {
        try {
            final PGPMimeDecryptionResult decryptionResult = pgpMimeCryptoService.decrypt(message, userIdentity);
            final MimeMessage decodedMsg = decryptionResult.getMimeMessage();
            final List<PGPSignatureVerificationResult> decrypt = decryptionResult.getSignatureVerificationResults();
            FromHeaderVerifier.verify(message, decrypt);
            checkMissingHeaders(message, decodedMsg);
            if (!decryptionResult.getMDCVerifiacionResult().isPresent()) {
                if (guardConfigService.getBooleanProperty(GuardProperty.failForMissingMDC)) {
                    throw MimeEncryptorExceptionCodes.MISSING_MDC.create();
                } else {
                    try {
                        decodedMsg.addHeader("X-Guard-Failed-Integrity", "TRUE");
                        new HTMLStripper().stripAllHTML(decodedMsg);
                    } catch (IOException e) {
                        throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
                    }
                }
            }
            decodedMsg.removeHeader("X-Guard-Signature-Result");
            String[] signatures = SignatureResults.toHeaders(decrypt);
            for (String sig : signatures) {
                decodedMsg.addHeader("X-Guard-Signature-Result", sig);
            }
            if (message.getHeader("X-Security") != null) {
                decodedMsg.addHeader("X-Security", message.getHeader("X-Security")[0]);
            }
            return decodedMsg;
        } catch (MessagingException e) {
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

}
