/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.milter.internal;

import java.awt.image.DataBuffer;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import org.slf4j.Logger;

/**
 * {@link MilterPacket}
 *
 * Creates a milter command packet, which has the command, data, and length
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class MilterPacket {

    private static Logger Log = org.slf4j.LoggerFactory.getLogger(MilterPacket.class);

    private int command;

    private int dataLength, dataRead, headerLength;

    private ByteBuffer header, data;

    private boolean headerDone, dataDone;


    public MilterPacket () {
        this.header = ByteBuffer.allocate(5);
        this.data = null;
        this.command = 0;
        this.dataLength = 0;
        this.dataRead = 0;
        this.headerLength = 0;
        this.headerDone = false;
        this.dataDone = false;
    }

    private static int uByteToInt(byte b)
    {
        return (((int) b) & 0x0FF);
    }

    /*
     * Get the header of the data packet.  Always 5 characters.  4 lenght, one command
     */
    private void getHeader (ByteBuffer dataBuffer) {
        int toRead = Math.min(5 - headerLength, dataBuffer.remaining());
        for (int i = 0; i < toRead; i++) {
            header.put(dataBuffer.get());
            headerLength++;
        }
        // If we have it all, parse the length and command
        if (headerLength == 5) {
            header.flip();
            headerDone = true;
            this.dataLength = processLength() - 1;  // Length - 1 for command
            this.command = processCommand();
            if (this.dataLength == 0) {
                this.dataDone = true;
                this.data = null;
            } else
                this.data = ByteBuffer.allocate(dataLength);  // Initiate data buffer
        }
    }

    /**
     * Get the data from the buffer.  Should be equal to the length specified in the header
     * This may come in multiple parts
     * @param dataBuffer
     */
    private void getData (ByteBuffer dataBuffer) {
        int readableLength = Math.min(dataLength - dataRead, dataBuffer.remaining());
        this.data.put((ByteBuffer) dataBuffer.asReadOnlyBuffer().limit(dataBuffer.position() + readableLength));
        dataBuffer.position(dataBuffer.position() + readableLength);
        dataRead += readableLength;
        if (dataRead == dataLength) {
            data.flip();
            try {
                byte[] dataArray = data.array();
                String stringData = new String(dataArray, "UTF-8").toString();
            } catch (UnsupportedEncodingException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
            dataDone = true;
        }

    }

    /**
     * Get the 4 byte length of data
     * @param dataBuffer
     * @return
     */
    private int processLength () {
        int length = 0;
        for (int i = 0; i < 4; i++) {
            length <<= 8;
            length += uByteToInt(header.get());
        }
        return length;
    }

    // Call after length
    private int processCommand () {
        int command = uByteToInt(header.get());
        this.command = command;
        return command;
    }

    public boolean process (ByteBuffer dataBuffer) {

        if (!headerDone) {
            getHeader (dataBuffer);
        }

        if (headerDone && !dataDone) {
            if (dataBuffer.hasRemaining()) {
                getData (dataBuffer);
            }
        }

        if (dataDone && headerDone) return true;

        return false;

    }

    public int getCommand () {
        return this.command;
    }

    public ByteBuffer getData () {
        return this.data;
    }

    public void reset() {
        this.header = ByteBuffer.allocate(5);
        this.data = null;
        this.command = 0;
        this.dataLength = 0;
        this.dataRead = 0;
        this.headerLength = 0;
        this.headerDone = false;
        this.dataDone = false;
    }
}
