/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.inputvalidation;

import java.util.Objects;

/**
 * {@link RangeInputValidator} validates that a {@link Number} is in a specific range.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class RangeInputValidator<T extends Number> extends AbstractInputValidator<T> {

    private final T min;
    private final T max;

    /**
     * Initializes a new {@link RangeInputValidator}.
     *
     * @param min The minimum value which is considered as valid.
     * @param max The maximum value which is considered as valid.
     */
    public RangeInputValidator(T min, T max) {
        min = Objects.requireNonNull(min, "min must not be null");
        max = Objects.requireNonNull(max, "max must not be null");
        this.min = min;
        this.max = max;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.inputvalidation.InputValidator#isValid(java.lang.Object)
     */
    @Override
    public boolean isValid(T input) {
        if(input == null) {
            return false;
        }
        if(input instanceof Byte) {
            return input.byteValue() >= min.byteValue() && input.byteValue() <= max.byteValue();
        }
        else if(input instanceof Integer) {
            return input.intValue() >= min.intValue() && input.intValue() <= max.intValue();
        }
        else if(input instanceof Float) {
            return input.floatValue() >= min.floatValue() && input.floatValue() <= max.floatValue();
        }
        else if(input instanceof Integer) {
            return input.intValue() >= min.intValue() && input.intValue() <= max.intValue();
        }
        else if(input instanceof Long) {
            return input.longValue() >= min.longValue() && input.longValue() <= max.longValue();
        }
        else if(input instanceof Short) {
            return input.shortValue() >= min.shortValue() && input.shortValue() <= max.shortValue();
        }
        else if(input instanceof Double) {
            return input.doubleValue() >= min.doubleValue() && input.doubleValue() <= max.doubleValue();
        }
        else {
            throw new IllegalArgumentException(String.format("type %s not supported", input.getClass().getCanonicalName()));
        }
    }
}
