/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.database;

import java.sql.Connection;
import java.util.List;
import com.openexchange.exception.OXException;
import com.openexchange.guard.database.dao.SchemaCandidate;

/**
 * The {@link DatabaseMaintenanceService} handles everything related to the creation of databases (existence, register in configdb, adding tables, ...)
 *
 * @author <a href="mailto:martin.schneider@open-xchange.com">Martin Schneider</a>
 * @since 2.4.0
 */
public interface DatabaseMaintenanceService {

    /**
     * Registers the schema (provided by the given {@link SchemaCandidate}) as a database within the configdb. If no slave URL is provided within the {@link SchemaCandidate} only master setup will be registered.
     * 
     * @param candidate {@link SchemaCandidate} instance holding all relevant information about master/slave handling
     * @throws OXException
     */
    void register(SchemaCandidate candidate) throws OXException;

    /**
     * Creates the schema based on the given information encapsulated within the {@link SchemaCandidate}.<br>
     * <br>
     * Note: This happens without prior checking of schema existence (see com.openexchange.guard.database.DatabaseMaintenanceService.exists(Connection, String))
     * 
     * @param candidate {@link SchemaCandidate} instance holding all relevant information about master/slave handling
     * @throws OXException
     */
    void createSchema(SchemaCandidate candidate) throws OXException;

    /**
     * Checks if the schema with the given name already exists.
     * 
     * @param connection The connection that should be used to check the existence
     * @param schemaName The name of the schema the existence should be checked for
     * @return boolean <code>true</code>, if the schema already exists; otherwise <code>false</code>
     * @throws OXException
     */
    boolean exists(Connection connection, String schemaName) throws OXException;

    /**
     * Creates schemas provided as {@link SchemaCandidate}s if they do not exist.<br>
     * <br>
     * Master/Slave schemas have to be created in one action. If no slave URL is provided within the {@link SchemaCandidate} only master setup will be registered.
     * 
     * @param candidates {@link List} of all {@link SchemaCandidate}s that should be created if not existent and registered within the configdb
     * @throws OXException if the schema already exists or there was an error while creation/registration
     */
    void createAndRegisterSchemaIfNotExistent(SchemaCandidate... candidates) throws OXException;
}
