/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.configuration.validation;

import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.inputvalidation.InputValidator;

/**
 * {@link ConfigurationValidator} validates configuration properties
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.2
 */
public class ConfigurationValidator {

    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(ConfigurationValidator.class);

    private final GuardConfigurationService configurationService;

    /**
     * Initializes a new {@link ConfigurationValidator}.
     * 
     * @param configurationService The configuration service
     */
    public ConfigurationValidator(GuardConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    @SuppressWarnings("unchecked")
    private <T> T getConfigurationValue(GuardProperty property, Class<? extends T> clazz) {
        T value = null;
        if (clazz.isAssignableFrom(Integer.class)) {
            return (T) new Integer(configurationService.getIntProperty(property));
        }
        else if (clazz.isAssignableFrom(Boolean.class)) {
            return (T) new Boolean(configurationService.getBooleanProperty(property));
        }
        else if (clazz.isAssignableFrom(String.class)) {
            return (T) configurationService.getProperty(property);
        }

        throw new IllegalArgumentException("Unknown property class type");
    }

    /**
     * Validates that a given GuardProperty is valid
     * 
     * @param property The property to validate
     * @param clazz The type of the property value
     * @param validator The validator used for validation
     * @param invalidHandler A handler which is called if the validation failed
     */
    public <T> void validateValue(GuardProperty property, Class<? extends T> clazz, InputValidator<T> validator, InvalidValueHandler<T> invalidHandler) {
        T value = getConfigurationValue(property, clazz);
        if (!validator.isValid(value)) {
            invalidHandler.handle(value);
        }
    }

    /**
     * Validates that a given GuardProperty is valid and throws an exception if it is not valid.
     * 
     * @param property The property to validate
     * @param clazz The type of the property value
     * @param validator The validator used for validation
     * @param invalidHandler A handler which is called if the validation failed
     */
    public <T> void assertValue(GuardProperty property, Class<? extends T> clazz, InputValidator<T> validator, InvalidValueHandler<T> invalidValueHandler) throws OXException {
        T value = getConfigurationValue(property, clazz);
        try {
            validator.assertIsValid(value);
        } catch (OXException e) {
            invalidValueHandler.handle(value);
            throw e;
        }
    }
}
