/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.common.servlets.utils;

import javax.servlet.http.HttpServletRequest;
import com.openexchange.antiabuse.AllowParameters;
import com.openexchange.antiabuse.Protocol;
import com.openexchange.antiabuse.ReportParameters;
import com.openexchange.antiabuse.ReportValue;

/**
 * {@link AntiAbuseUtils}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class AntiAbuseUtils {

    /**
     * Overload method to create allow Parameter from userid /cid combination
     * @param request
     * @param userid
     * @param cid
     * @param password
     * @return
     */
    public static AllowParameters getAllowParameters (HttpServletRequest request, int userid, int cid, String password) {
        return getAllowParameters (request, userid + "-" + cid, password);
    }

    /**
     * Create antiabuse AllowParmeter from HttpServlet Request
     * @param request
     * @param login
     * @param password
     * @return
     */
    public static AllowParameters getAllowParameters (HttpServletRequest request, String login, String password) {
        return getAllowParameter (
            login,
            password,
            ServletUtils.getClientIP(request),
            request.getHeader("User-Agent"),
            request.getScheme().toLowerCase().contains("https") ? Protocol.HTTPS : Protocol.HTTP
            );
    }

    /**
     * Create antiabuse AllowParameter from elements
     * @param login
     * @param password
     * @param ip
     * @param userAgent
     * @param protocol
     * @return
     */
    public static AllowParameters getAllowParameter (String login, String password, String ip, String userAgent, Protocol protocol) {
        AllowParameters.Builder parameters = AllowParameters.builder()
            .login(login)
            .password(password)
            .remoteAddress(ip)
            .userAgent(userAgent)
            .protocol(protocol);
        return (parameters.build());
    }

    /**
     * Overload method to create Report Parameter from http request using userid and cid
     * @param success
     * @param request
     * @param userId
     * @param cid
     * @param password
     * @return
     */
    public static ReportParameters getReportParameter (boolean success, HttpServletRequest request, int userId, int cid, String password) {
        return getReportParameter (success, request, userId + "-" + cid, password);
    }

    /**
     * Overload method to create report Parameter from Http Request object
     * @param success
     * @param request
     * @param login
     * @param password
     * @return
     */
    public static ReportParameters getReportParameter (boolean success, HttpServletRequest request, String login, String password) {
        return getReportParameter (
            success ? ReportValue.SUCCESS : ReportValue.FAILURE,
                login,
                password,
                ServletUtils.getClientIP(request),
                request.getHeader("User-Agent"),
                request.getScheme().toLowerCase().contains("https") ? Protocol.HTTPS : Protocol.HTTP);
    }

    /**
     * Overload method to create report from AllowParameter
     * @param reportValue
     * @param allowParam
     * @return
     */
    public static ReportParameters getReportParameter (ReportValue reportValue, AllowParameters allowParam) {
        return getReportParameter (reportValue,
            allowParam.getLogin(),
            allowParam.getPassword(),
            allowParam.getRemoteAddress(),
            allowParam.getUserAgent(),
            allowParam.getProtocol());
    }

    /**
     * Create ReportParameter from individual values
     * @param reportValue
     * @param login
     * @param password
     * @param remoteAddress
     * @param userAgent
     * @param protocol
     * @return
     */
    public static ReportParameters getReportParameter (ReportValue reportValue, String login, String password, String remoteAddress, String userAgent, Protocol protocol) {
        ReportParameters.Builder parameters = ReportParameters.builder()
            .reportValue(reportValue)
            .login(login)
            .password(password)
            .remoteAddress(remoteAddress)
            .protocol(protocol)
            .userAgent(userAgent);
        return parameters.build();
    }

}
