define('pgp_mail/toggle-encryption', [
    'io.ox/core/capabilities',
    'gettext!oxguard',
    'settings!oxguard',
    'less!pgp_mail/style'
], function (capabilities, gt, settings) {

    'use strict';

    var ToggleEncryptionView = Backbone.View.extend({
        tagName: 'button',
        initialize: function () {
            this.listenTo(this.model, 'change:encrypt', function (model, val) {
                this.$('i.fa').toggleClass('encrypted', val);
                //gt('Click to enable encryption')
                //gt('Click to disable encryption')
            });
        },
        className: 'toggle-encryption',
        events: {
            'click': 'toggle',
            'change:encrypt': 'changed'
        },
        toggle: function () {
            if (this.encryption_forced) {
                require (['io.ox/core/notifications'], function (notify) {
                    notify.yell('error', gt('Reply must be encrypted'));
                });
            } else {
                this.model.set('encrypt', !this.model.get('encrypt'));
            }
        },
        forceEncryption: function () {
            this.encryption_forced = true;
            this.model.set('encrypt', true);
            this.render();
        },
        render: function () {
            var guard = false;
            if (window.oxguarddata !== undefined) {
                if (window.oxguarddata.settings !== undefined) {
                    guard = (window.oxguarddata.settings.oxguard === true);
                }
            }
            if (guard || this.encryption_forced)
                {
                if (!_.device('small')) {
                    this.$el.empty().append(
                            $('<i class="fa fa-2x" aria-hidden="true">')
                            .toggleClass('encrypted', !!this.model.get('encrypt'))
                        );
                    this.$el.attr('title', gt('Enable Encryption'));
                }
            }
            this.listenTo(this.model, 'change:encrypt', function () {
                this.changed();
            });
            if (this.model.get('encrypt')) {
                var view = this.view;
                window.setTimeout(function () {
                    showAttLink(view, false);
                }, 100);
            }
            return this;
        },
        noLinkMail: function (view) {
            this.view = view;
            checkAttLink (this.model, view);
        },
        changed: function () {
            var security = _.extend({}, this.model.get('security'));
            security.encrypt = this.model.get('encrypt');
            this.model.set('security', security);
            this.$el.attr('title', security.encrypt ? gt('Disable Encryption') : gt('Enable Encryption'));
            if (this.model && this.model.get('encrypt')) {
                showAttLink(this.view, false);
                checkAttLink(this.model);
                try {
                    var fromArray = this.model.get('from');
                    if (fromArray && window.oxguarddata.primaryEmail !== undefined) {
                        if (window.oxguarddata.primaryEmail !== fromArray[0][1]) {
                            var warn = settings.get('warnEmailDifferent');
                            if (warn === undefined || warn === false) {
                                displayWarning();
                            }
                        }
                    }
                } catch (e) {
                    console.log(e);
                }
            } else {
                showAttLink(this.view, true);
            }
        }
    });

    // Hide or show mail attachment link
    function showAttLink (view, show) {
        if (view) {
            view.$el.find('.share-attachments').toggleClass('guardDisabled', !show);
        }
    }

    // Check if Mail Attachment link is shown and encryption enabled
    function checkAttLink (model, view) {
        if (model.get('share_attachments')) {
            if (model.get('share_attachments').enable) {
                if (model.get('encrypt')) {
                    showAttLink(view, false);
                }
            }
        }
    }

    // Show a warning if Mail Attachment Link is shown and encryption clicked
    function checkAttLink (model) {
        if (model.get('share_attachments')) {
            if (model.get('share_attachments').enable) {
                require (['io.ox/core/tk/dialogs', 'settings!io.ox/mail'], function (dialogs, mail) {
                    var dialog = new dialogs.CreateDialog({ width: 450, height: 300, center: true, enter: 'ok' });
                    dialog.header($('<h4>').text(gt('Not Supported')));
                    var text = $('<p>').text(gt('%s is not supported with secured email and will be disabled.', mail.get('compose/shareAttachments/name')));
                    dialog.getBody().append(text);
                    dialog
                    .addPrimaryButton('ok', gt('OK'), 'ok')
                    .on('ok', function () {
                        $('.share-attachments').find(':checkbox').prop('checked', false);
                        var share = model.get('share_attachments');
                        share.enable = false;
                        model.set('share_attachments', share);
                    })
                    .show();
                });
            }
        }
    }

    // Warning that the from address is different from primary
    function displayWarning () {
        require (['io.ox/core/tk/dialogs'], function (dialogs) {
            var dialog = new dialogs.CreateDialog({ width: 450, height: 300, center: true, enter: 'ok' });
            dialog.header($('<h4>').text(gt('Warning')));
            var text = $('<p>').text(gt('Sending an encrypted email from an account other than %s may cause problems with the recipient being able to reply.  Consider using your primary account.', window.oxguarddata.primaryEmail));
            var warn = $('<input type="checkbox" id="warnEmail">');
            var warntext = $('<span class="selectable" style="padding-left:10px;">').text(gt('Do not warn me again'));
            warntext.click(function () {
                warn.click();
            });
            warn.click(function (e) {
                settings.set('warnEmailDifferent', $(e.target).is(':checked'));
                settings.save();
            });
            dialog.getBody().append(text).append('<hr>').append(warn).append(warntext);
            dialog
            .addPrimaryButton('ok', gt('OK'), 'ok')
            .show();
        });
    }

    return {
        View: ToggleEncryptionView
    };
});
