/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 * Author Greg Hill <greg.hill@open-xchange.com>
 */
define('pgp_local/storage', [
], function () {
    'use strict';

    var indexedDB = window.indexedDB || window.mozIndexedDB || window.webkitIndexedDB || window.msIndexedDB;

    function open () {
        var def = $.Deferred();
        var request = indexedDB.open('guard', 1);
        var db;
        request.onerror = function (event) {
            console.log('Error opening DB', event);
            def.reject();
        };
        request.onupgradeneeded   = function (event) {
            db = event.target.result;
            var transaction = event.target.transaction;
            transaction.oncomplete = function () {
                def.resolve(db);
            };
            db.createObjectStore('guardKeys', { keyPath: 'id' });
        };
        request.onsuccess  = function (event) {
            db = event.target.result;
            def.resolve(db);
        };
        return def;
    }

    function close (db) {
        db.close();
    }

    function save (db, id, data) {
        var def = $.Deferred();
        var transaction = db.transaction(['guardKeys'], 'readwrite');
        transaction.oncomplete = function () {
            def.resolve();
        };
        transaction.onerror = function () {
            def.reject();
        };
        var objectStore = transaction.objectStore('guardKeys');
        objectStore.put({
            id: id,
            data: data
        });
        return def;
    }

    function get (db, id) {
        var def = $.Deferred();
        var request = db.transaction(['guardKeys'], 'readonly').objectStore('guardKeys').get(id);
        request.onsuccess = function () {
            if (!request.result) {
                def.reject();
                return;
            }
            def.resolve(request.result.data);
        };
        return def;
    }

    function deleteKey (db, id) {
        var def = $.Deferred();
        var request = db.transaction(['guardKeys'], 'readwrite').objectStore('guardKeys').delete(id);
        request.onsuccess = function () {
            def.resolve();
        };
        request.onerror = function () {
            def.reject();
        };
        return def;
    }

    var storage = {
        get: function (id) {
            var def = $.Deferred();
            open()
            .done(function (db) {
                get(db, id)
                .done(function (result) {
                    def.resolve(result);
                })
                .fail(function (e) {
                    console.log(e);
                    def.reject();
                })
                .always(function () {
                    close(db);
                });
            });
            return def;
        },
        save: function (id, data) {
            open()
            .done(function (db) {
                save(db, id, data)
                .always(function () {
                    close(db);
                });
            });
        },
        deleteKeys: function (id) {
            open()
            .done(function (db) {
                deleteKey(db, id)
                .always(function () {
                    close(db);
                });
            });
        }
    };

    return storage;

});
