/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Author Greg Hill <greg.hill@open-xchange.com>
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('pgp_local/session', [
    'pgp_local/base64'
], function (base64) {
    'use strict';

    var openpgp = window.openpgp;
    var keyRing = window.keyRing;

    function convert (data) {
        return base64.decode(data);
    }

    function keyIdCheck (id1, key) {
        return id1.bytes == key.keyid.bytes;
    }

    function getForId (privkeys, keyId) {
        for (var i = 0; i < privkeys.keys.length; i++) {
            privkeys.keys[i].getKeyIds();
            if (keyIdCheck(keyId, privkeys.keys[i].primaryKey)) {
                return privkeys.keys[i];
            }
            if (privkeys.keys[i].subKeys) {
                for (var j = 0; j < privkeys.keys[i].subKeys.length; j++) {
                    if (keyIdCheck(keyId, privkeys.keys[i].subKeys[j].subKey)) {
                        return privkeys.keys[i];
                    }
                }
            }
        }
        return null;
    }

    function getCheckSum (array) {
        var check = 0;
        var c = [];
        for (var i = 0; i < array.length; i++) {
            check += array[i];
        }
        c.push((check & 0xFF00) >> 8);
        c.push(check & 0xFF);
        return new Uint8Array(c);
    }

    var session = {
        setSession: function (value) {
            this.pgpSession = new openpgp.packet.PublicKeyEncryptedSessionKey();
            var value = convert(value);
            this.pgpSession.read(value.slice(3));
        },
        hasKey: function () {
            var keyId = this.pgpSession.publicKeyId;
            return (getForId(keyRing.privateKeys, keyId) !== null);
        },
        decrypt: function (password) {
            var keyId = this.pgpSession.publicKeyId;
            var privateKey = getForId(keyRing.privateKeys, keyId);
            if (privateKey === null) return 'noKey';
            if (password) {

            }
            if (!privateKey.decrypt(password)) return 'badPassword';
            var key = privateKey.getKeyPacket([keyId]);
            if (!key.isDecrypted) {
                return 'notDecrypted';
            }
            if (key === null || key === undefined) return 'no key';
            this.pgpSession.decrypt(key);
            key.encrypt(password);  // Re-encrypt private key and wipe private data
            key.clearPrivateMPIs();
            return 'ok';
        },
        getDecrypted: function () {
            var checked = new Uint8Array (this.pgpSession.sessionKey.length + 3);
            var alg = window.openpgp.enums.symmetric[this.pgpSession.sessionKeyAlgorithm];
            checked.set([alg]);
            checked.set(this.pgpSession.sessionKey, 1);
            checked.set(getCheckSum(this.pgpSession.sessionKey), this.pgpSession.sessionKey.length + 1);
            var b64 = base64.encode(checked);
            return b64;
        }

    };

    return session;
});
