/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

define('oxguard/settings/views/changePasswordView', [
    'io.ox/core/extensions',
    'io.ox/backbone/views/modal',
    'io.ox/core/yell',
    'oxguard/core/og_http',
    'oxguard/oxguard_core',
    'oxguard/core/passwordView',
    'gettext!oxguard'
], function (ext, ModalView, yell, og_http, core, PasswordView, gt) {

    'use strict';

    var POINT = 'oxguard/settings/passwords',
        INDEX = 0;

    function open (node, baton) {
        return openModalDialog(node, baton);
    }

    function openModalDialog(node, baton) {

        return new ModalView({
            async: true,
            focus: 'input[name="active"]',
            point: POINT,
            title: gt('Change Password'),
            width: 640
        })
        .inject({
            doChange: function () {
                var newpass1 = $('#newpass1');
                var newpass2 = $('#newpass2');
                var oldpass = $('#oldpass');
                return changePass(newpass1, newpass2, oldpass);
            }
        })
        .build(function () {

        })
        .addCancelButton()
        .addButton({ label: gt('Change'), action: 'change' })
        .on('change', function () {
            var view = this;
            this.doChange().done(function () {
                view.close();
                ext.point('oxguard/settings/detail').invoke('draw', node, baton);  // redraw settings
            }).fail(this.idle).fail(yell);
        })
        .open();
    }

    ext.point(POINT).extend(
            //
            // Password Boxes
            //
            {
                index: INDEX += 100,
                id: 'switch',
                render: function () {

                    this.$body.append(
                        generatePasswordPrompts()
                    );
                }
            }
     );

    function generatePasswordPrompts () {
        var passheader =
            $('<div class="oxguard_settings password"/>');
        var oldpass = new PasswordView.view({ 'id': 'oldpass', 'class': 'password_prompt' }).getInputField();
        var newpass1 = new PasswordView.view({ 'id': 'newpass1', 'class': 'password_prompt', 'validate': true });
        var newpass2 = new PasswordView.view({ 'id': 'newpass2', 'class': 'password_prompt' });
        var td = $('<td>').append(newpass1.getInputField());
        var td2 = $('<td>').append(newpass2.getInputField());
        var hint = $('<td style="width:150px;">');
        var curPrompt = $('<label for="oldpass">').append(gt('Enter current %s security password', window.oxguarddata.productName));
        var prompt1 = $('<label for="newpass1">').append(gt('Enter new %s security password', window.oxguarddata.productName));
        var prompt2 = $('<label for="newpass2">').append(gt('Verify new %s security password', window.oxguarddata.productName));
        var passwordTable;
        if (_.device('small')) {
            passwordTable = $('<div style="padding:10px;">');
            if (!window.oxguarddata['new']) {
                passwordTable        //#. %s product name
                .append(curPrompt).append('<br>').append(oldpass).append('<br>');
            }
            passwordTable
            .append(prompt1).append('<br>').append(newpass1.getProtected()).append('<br>')
            .append(prompt2).append('<br>').append(newpass2.getProtected);
        } else {
            passwordTable = $('<table/>');
            if (!window.oxguarddata['new']) {
                passwordTable        //#. %s product name
                .append($('<tr>').append(($('<td>').append(curPrompt))).append($('<td>').append(oldpass)));
            }
            passwordTable
            .append($('<tr>').append(($('<td>').append(prompt1))).append(td).append(hint))
            .append($('<tr>').append(($('<td>').append(prompt2))).append(td2).append($('<td>')));
        }

        if (window.oxguarddata.pin) {
            var pinPrompt = $('<p>').append(gt('The sender assigned an additional PIN to the account.  This PIN must be entered before you can use this account.  Please enter it now.  Contact the sender directly if you don\'t yet have a pin.'));
            var pinTable = $('<table class="og_password_prompt">');
            var pinLabel = $('<td class="pw">').append('<label for="pinPrompt">' + gt('PIN') + ':</label>');
            var pinEntry = $('<input id="pinInput" name="pinInput" class="form-control password_prompt">');
            passheader.append(pinPrompt).append(pinTable.append($('<tr>').append(pinLabel).append($('<td>').append(pinEntry)))).append('<hr/>');
        }

        require (['oxguard/core/passwords'], function (pass) {
            pass.passwordCheck(newpass1, hint);
            pass.passwordCompare(newpass2, newpass1, hint);
        });
        return passheader.append(passwordTable);
    }

    function changePass(newpass, newpass2, oldpass) {
        var def = $.Deferred();
        var min = window.oxguarddata.settings.min_password_length;
        if (min !== undefined) {
            if (newpass.val().length < min) {
                showError(gt('New password must be at least %s characters long', min));
                if (!_.device('ios')) {
                    newpass.focus();
                }
                def.reject();
                return def;
            }
        }
        if (newpass.val() !== newpass2.val()) {
            newpass2.css('background-color', 'salmon');
            def.reject();
            return def;
        } else {
            newpass2.css('background-color', 'white');
            var userdata = {
                    newpass: newpass.val(),
                    oldpass: oldpass.val(),
                    sessionID: ox.session,
                    cid: ox.context_id,
                    userEmail: ox.user,
                    pin: window.oxguarddata.pin ? $('#pinInput').val() : ''
                };
            var waitdiv = $('<div class="og_wait" id="keygen"></div');
            waitdiv.append('<i class="fa-key icon-key fa fa-spin icon-spin"/><span>' + gt('Generating key, Please wait') + '</span>');
            $('.window-container-center').append(waitdiv);
            waitdiv.show();
            core.metrics('settings', 'change-password');
            og_http.post(ox.apiRoot + '/oxguard/login?action=changepass', '', userdata)
            .done(function (data) {
                $('.og_wait').remove();
                if (typeof data === 'string') data = $.parseJSON(data);
                if (data.auth === undefined) {
                    require(['oxguard/core/errorHandler'], function (err) {
                        err.showError(data);
                    });
                    def.reject();
                    return;
                }
                if (data.auth === 'Bad password') {
                    showError(gt('Bad password'));
                }
                if (data.auth === 'Lockout') {
                    showError(gt('Temporary Lockout'));
                }
                if (data.auth === 'Bad new password') {
                    showError(gt('New password must be at least %s characters long', data.minlength));
                }
                if (data.auth.length > 20) {
                    $('input[name="newpass1"]').val('');
                    $('input[name="newpass2"]').val('');
                    $('input[name="oldpass"]').val('');
                    window.oxguarddata.passcode = null;
                    window.oxguarddata.pin = false;
                    window.oxguarddata.recovery = data.recovery;
                    window.oxguarddata['new'] = false;
                    showOK(gt('Password changed successfully'));
                    core.removeAuth(); // We want to remove rembered auth codes, as the password has changed
                    def.resolve();
                }
            })
            .fail(function (e) {
                $('.og_wait').remove();
                var error = gt('Failed to change password');
                if (e.responseText.trim() === 'Bad Password') {
                    error = gt('Bad password');
                }
                showError(error);
                def.reject();
            });
            return def;
        }
    }

    function showOK (msg) {
        require(['io.ox/core/notifications'], function (notify) {
            notify.yell('success', msg);
        });
    }

    function showError (msg) {
        require(['io.ox/core/notifications'], function (notify) {
            notify.yell('error', msg);
        });
    }

    return {
        open: open
    };

});
