/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/settings/pane', [
    'io.ox/core/extensions',
    'io.ox/backbone/views/extensible',
    'io.ox/core/settings/util',
    'oxguard/util',
    'io.ox/core/notifications',
    'gettext!oxguard',
    'oxguard/oxguard_core',
    'settings!oxguard',
    'less!oxguard/settings/style'
], function (ext, ExtensibleView, util, guardUtil, notifications, gt, core, settings) {
    'use strict';

    ext.point('oxguard/settings/detail').extend({
        index: 100,
        id: 'oxguardsettings',
        draw: function (baton) {
            var node = this;
            if (guardUtil.isGuardLoaded()) {
                drawGuardSettings(node, baton);
            } else {
                guardUtil.addOnLoaded(function () {
                    drawGuardSettings(node, baton);
                });
            }
        }
    });

    var INDEX = 0;

    function drawGuardSettings (node, baton) {
        guardUtil.hasSetupDone()
        .done(function () {
            node.empty();
            node.addClass('guard-settings');
            node.append(
                    new ExtensibleView({ point: 'oxguard/settings/detail/view', model: settings })
                    .inject({
                        getPasswordOptions: function () {
                            return ([
                                { label: gt('Ask each time'), value: 0 },
                                { label: gt('10 minutes'), value: 10 },
                                { label: gt('20 minutes'), value: 20 },
                                { label: gt('30 minutes'), value: 30 },
                                { label: gt('1 hour'), value: 60 },
                                { label: gt('2 hours'), value: 120 },
                                { label: gt('Session'), value: 99999 }
                            ]);
                        }
                    })
                    .build(function () {
                        this.listenTo(settings, 'change', function () {
                            settings.saveAndYell().then(
                                function ok() {
                                    //
                                },
                                function fail() {
                                    notifications.yell('error', gt('Could not save settings'));
                                }
                            );
                        });
                    })
                    .render().$el
                );
        })
        .fail (function () {
            require(['io.ox/core/capabilities'], function (capabilities) {
                if (capabilities.has('guard-mail') || capabilities.has('guard-drive')) {
                    node.append(setupGuard(node, baton));
                }
            });
            return;
        });
    }

    ext.point('oxguard/settings/detail/view').extend(
            {
                id: 'header',
                index: INDEX += 100,
                render: function () {
                    this.$el.addClass('io-ox-guard-settings').append(
                        util.header(gt('%s security settings', window.oxguarddata.productName))
                    );
                }
            },
            {
                id: 'defaults',
                index: INDEX += 100,
                render: function () {
                    if (guardUtil.isGuest() || !guardUtil.hasGuardMailCapability()) return;   // Guests don't set defaults for sending
                    this.$el.append(
                        $('<div class="guardDefaults">').append(
                            util.fieldset(
                                gt('Defaults'),
                                // default to encrypt outgoing
                                util.checkbox('defaultEncrypted', gt('Default to send encrypted when composing email'), settings),
                                // default add signature
                                util.checkbox('defaultSign', gt('Default adding signature to outgoing mails'), settings),
                                // inline
                                settings.get('advanced', false) ?
                                    util.checkbox('defaultInline', gt('Default to using PGP inline for new mails'), settings) : [],
                                util.compactSelect('defRemember', gt('Remember password default'), settings, this.getPasswordOptions())
                            )
                        )
                    );
                }
            },
            {
                id: 'passwordManagement',
                index: INDEX += 100,
                render: function (baton) {
                    this.$el.append(
                        util.fieldset(
                            gt('Password management'),
                            baton.branch('buttons', null, $('<div class="form-group buttons" id="passwordDiv">'))
                        )
                    );
                }
            },
            {
                id: 'Advanced',
                index: INDEX += 100,
                render: function () {
                    if (!guardUtil.hasGuardMailCapability()) return;
                    var advancedBox = util.checkbox('advanced', gt('Show advanced settings'), settings);
                    advancedBox.on('change', function () {
                        updateAdvanced();
                    });
                    this.$el.append(
                        util.fieldset(
                            gt('Advanced'),
                            // default to encrypt outgoing
                            advancedBox
                        )
                    );
                    this.$el.append(drawAdvanced());
                }
            }
        );
    //
    // Buttons
    //
    ext.point('oxguard/settings/detail/view/buttons').extend(
        {
            id: 'changePassword',
            index: 100,
            render: function (baton) {
                this.append(
                    $('<button type="button" class="btn btn-default" id="changePassword">')
                    .text(gt('Change password'))
                    .on('click', openDialog)
                );

                function openDialog() {
                    ox.load(['oxguard/settings/views/changePasswordView']).done(function (view) {
                        var node = $('.guard-settings');
                        view.open(node, baton);
                    });
                }
            }
        },
        {
            id: 'resetPassword',
            index: 200,
            render: function () {

                // check enabled
                if (!window.oxguarddata.recovery || window.oxguarddata['new']) {  // If no recovery avail, then don't display reset button
                    return;
                }
                this.append(
                    $('<button type="button" class="btn btn-default">')
                    .text(gt('Reset password'))
                    .on('click', openDialog)
                );

                function openDialog() {
                    ox.load(['oxguard/settings/views/resetPasswordView']).done(function (view) {
                        view.open();
                    });
                }
            }
        },
        {
            id: 'changeEmail',
            index: 300,
            render: function () {

                // check enabled
                if (!window.oxguarddata.recovery || window.oxguarddata['new']) {  // If no recovery avail, then don't display reset button
                    return;
                }
                this.append(
                    $('<button type="button" class="btn btn-default">')
                    .text(gt('Set email address for reset'))
                    .on('click', openDialog)
                );

                function openDialog() {
                    ox.load(['oxguard/settings/views/secondaryEmailView']).done(function (view) {
                        view.open();
                    });
                }
            }
        },
        {
            id: 'deleteRecovery',
            index: 400,
            render: function (baton) {
                // check enabled
                if (!window.oxguarddata.recovery || window.oxguarddata.settings.noDeleteRecovery || window.oxguarddata['new']) {  // If no recovery avail, then don't display reset button
                    return;
                }
                var button = $('<button type="button" class="btn btn-default" id="deleteRecovery">')
                    //#. Option to remove the possibility of recovering your password
                    .text(gt('Delete password recovery'))
                    .on('click', openDialog);

                if (!settings.get('advanced')) {   // Only visible if advanced checked
                    button.css('display', 'none');
                }

                this.append(button);
                function openDialog() {
                    ox.load(['oxguard/settings/views/deleteRecoveryView']).done(function (view) {
                        var node = $('.guard-settings');
                        view.open(node, baton);
                    });
                }
            }
        }
    );

    function drawAdvanced () {
        var div = $('<div class="guardAdvanced" id="advancedDiv">');
        if (!settings.get('advanced')) {
            return div;
        }
        $('#deleteRecovery').show();
        var downloadPublic = $('<div>');
        var downloadPublicButton = $('<button type="button" class="btn btn-default" name="downloadPublic">')
        .text(gt('Download my public key'))
        .on('click', downloadPublicKey);
        downloadPublic.append(downloadPublicButton);
        var userKeys = $('<button type="button" class="btn btn-default" name="yourKeys">')
        .text(gt('Your keys'))
        .on('click', openUserKeysDialog);
        function openUserKeysDialog() {
            ox.load(['oxguard/settings/views/userKeysView']).done(function (view) {
                view.open();
            });
        }
        var recipients = '';
        if (!guardUtil.isGuest()) {  // Guests don't manage Public keys for recipients
            recipients = $('<button type="button" class="btn btn-default" name="recipients">')
            .text(gt('Public keys of recipients'))
            .on('click', openDialog);
        }
        function openDialog() {
            ox.load(['oxguard/settings/views/publicKeysView']).done(function (view) {
                view.open();
            });
        }
        div.append(util.fieldset(
                gt('Keys'), downloadPublic, userKeys, recipients
                ));
        div.append(drawVersion());
        return div;
    }

    function downloadPublicKey () {
        require(['oxguard/settings/views/downloadPrivateView'], function (downloader) {
            downloader.downloadPublic();
        });
    }

    function drawVersion () {
        if (_.device('small')) return ('');
        var div = $('<div class="guardVersion">');
        div.append(window.oxguarddata.version + '<br>build: ' + window.oxguarddata.server);
        return (div);
    }

    function updateAdvanced () {
        if (!settings.get('advanced')) {
            $('#deleteRecovery').hide();
            $('#advancedDiv').hide();
            return;
        }
        $('#advancedDiv').show();
        $('#advancedDiv').html(drawAdvanced());

    }

    // Convert old Guard server settings to local.  Should be able to remove version after 7.10.  Remove call from oxguard/register
    function initSettings(data) {
        settings.set('defaultEncrypted', data.pgpdefault === true);
        settings.set('defaultSign', data.pgpsign === true);
        settings.set('defaultInline', data.inline === true);
        settings.save();
    }

    function setupGuard (node, baton) {
        var div = $('<div>').addClass('io-ox-guard-settings');
        var passheader = util.header(gt('%s security settings', window.oxguarddata.productName));
        var instructions = $('<div>');
        //#. product name
        instructions.append('<p>' + gt('In order to start with %s security, you must set up a password for your secured files.', window.oxguarddata.productName) + '</p>');
        var startButton = $('<button type="button" class="btn btn-default guard-start-button">' + gt('Start') + '</button>');
        startButton.click (function () {
            require(['oxguard/core/createKeys'], function (keys) {
                core.metrics('settings', 'create-keys');
                keys.createKeysWizard()
                .done(function (result) {
                    if (result === 'OK') {
                        ext.point('oxguard/settings/detail').invoke('draw', node, baton);
                    }
                    //$('.oxguard_settings').replaceWith(addSettings());
                });
            });
        });
        return (div.append(passheader).append(instructions).append(startButton));
    }

    return {
        initSettings: initSettings
    };
});
