/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

define('oxguard/pgp/signatureView', [
    'io.ox/backbone/views',
    'io.ox/core/extensions',
    'io.ox/backbone/views/modal',
    'oxguard/core/og_http',
    'io.ox/core/yell',
    'gettext!oxguard',
    'less!oxguard/pgp/style'
], function (views, ext, ModalView, http, yell, gt) {

    'use strict';

    var POINT = 'oxguard/pgp/signatureview',
    INDEX = 0;

    function open (keyid) {
        return loadData(keyid).then(function (data) {
            openModalDialog(data);
        });
    }

    function openModalDialog(data) {

        return new ModalView({
            async: true,
            point: POINT,
            title: gt('Public Keys Detail'),
            id: 'pkeyDetail',
            width: 500,
            model: new Backbone.Model({ 'data': data })
        })
        .inject({

        })
        .addButton({ label: gt('Done'), action: 'done' })
        .on('done', function () {
            this.close();
        })
        .open();
    }

    ext.point(POINT).extend(
            // Draw signature
            {
                index: INDEX += 100,
                id: 'detail',
                render: function (baton) {
                    var data = baton.model.get('data');
                    var table = $('<table class="signatureTable">');
                    if (data.length === 0) {
                        table = $('<b>' + gt('No signatures') + '</b>');
                    } else {
                        //#. Table headers for the ID "signed", the level of the "certification", and who it was "signed by"
                        table.append($('<th>' + gt('ID Signed') + '</th><th>' + gt('Certification') + '</th><th>' + gt('Signed by') + '</th>'));
                        for (var i = 0; i < data.length; i++) {
                            var tr = $('<tr>');
                            var td1 = $('<td>');

                            if (data[i].ID === 'image') {
                                if (data[i].image !== undefined) {
                                    var image = $('<img alt="Embedded Image" src="' +  data[i].image + '" style="max-height:50px;max-width:50px;"/>');
                                    td1.append(image);
                                } else {
                                    //#. Type of object signed was an image
                                    data[i].ID = gt('Image');  // Translate image response
                                    td1.append(data[i].ID);
                                }
                            } else {
                                td1.append ($('<span>' + data[i].ID.replace(/>/gi,'&gt;').replace(/</gi,'&lt;') + '</span>'));
                            }
                            tr.append(td1);
                            var td2 = $('<td>');
                            td2.append (translate(data[i].result));
                            tr.append(td2);
                            var td3 = $('<td>');
                            td3.append (data[i].certifier.replace(/>/gi,'&gt;').replace(/</gi,'&lt;'));
                            tr.append(td3);
                            table.append(tr);
                        }
                    }
                    this.$body.append(
                        table
                    );
                }
            }
     );

    function translate (result) {
        switch (result) {
        case 'Positive':
            //#. In table, the key was "Positively" identified by the signer (verification level)
            return (gt('Positive'));
        case 'Fail':
            //#. In table, the signature failed to be verified
            return (gt('Failed check'));
        case 'Missing Public key':
            //#. In table, there was no public key available to verify the signature
            return (gt('Missing public'));
        case 'Casual':
            //#. In table, the key was "Casually" identified by the signer (verification level)
            return (gt('Casual'));
        case 'Revoke Cert':
            //#. In table, the key was revoked with this signature
            return (gt('Revocation'));
        case 'Direct':
            //#. In table, the key was "Directly" certified by the signer (verification level)
            return (gt('Direct cert'));
        case 'Revoke':
            //#. In table, the key was revoked with this signature
            return (gt('Revocation'));
        case 'Default Cert':
            //#. In table, the key had "Default certification" by the signer (verification level)
            return (gt('Default Certification'));
        case 'Sub revoke':
            //#. In table, the key was revoked with this signature
            return (gt('Revocation'));
        case 'Subkey Binding':
            //#. In table, the key is a bound subkey
            return (gt('Subkey Binding'));
        case 'Primary Binding':
            //#. In table, primary key binding
            return (gt('Primary Binding'));
        default:
            return (result);
        }
    }

    function loadData (keyId) {
        var def = $.Deferred();
        var params = '&userid=' + ox.user_id + '&cid=' + ox.context_id + '&keyid= ' + keyId;
        // Send the request to the oxguard server for the encrypted content key to decrypt
        var url = '/oxguard/pgp?action=getsignatures';
        http.get(ox.apiRoot + url, params)
        .done (function (data) {
            data = JSON.parse(data);
            def.resolve(data);
        })
        .fail (function () {
            require(['io.ox/core/notifications'], function (notify) {
                notify.yell('error', gt('Problem getting signatures'));
            });
            def.reject();
        });
        return def;
    }

    return {
        open: open
    };
});
