/* All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/pgp/register', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'io.ox/core/extPatterns/actions',
    'oxguard/core/og_http',
    'oxguard/mail/checker',
    'io.ox/core/api/account',
    'io.ox/core/capabilities',
    'pgp_mail/util',
    'oxguard/mail/mail_metrics',
    'io.ox/mail/common-extensions',
    'settings!oxguard',
    'gettext!oxguard',
    'oxguard/mail/mailReplyActions',
    'less!oxguard/pgp/style'
], function (ext, links, actions, http, checker, account, capabilities, util, metrics, mailExtensions, settings, gt) {
    'use strict';

    // Draw security icons in mail detail view
    mailExtensions.security = function (baton) {
        if (!baton.model.get('security')) return;
        var security = baton.model.get('security');
        var pgpclass = 'icon-lock fa fa-lock pgp_superscript';
        if (security.decrypted) {
            var pgp = $('<span class="oxguard_icon_fa" title="' + gt('This Email was encrypted with PGP.') + '"/>');
            pgp.append('P<span style="font-size:0.5em;">gp</span>');
            var lockicon = $('<i class="' + pgpclass + '" aria-hidden="true"/>');
            pgp.append(lockicon);
            if (_.device('small')) {
                pgp.css('margin-right', '5px');
            }
            this.append(pgp);
        }
        if (security.signatures) {
            var sig = security.signatures[0];
            var signedicon;
            if (sig.verified) {
                signedicon = $('<i class="oxguard_icon_fa fa-pencil-square-o fa guard_signed" aria-hidden="true" title="' + gt('This Email was signed and verified.') + '"/>');
            } else {
                if (!sig.missing) {
                    var display = gt('This Email was signed and failed verification.');
                    if (sig.error) {
                        display += ' ' + sig.error;
                    }
                    signedicon = $('<i class="oxguard_icon_fa_alert fa-pencil-square fa guard_signed" aria-hidden="true" title="' + display + '"/>');
                }
            }
            if (sig.missing) {
                signedicon = $('<i class="oxguard_icon_fa fa-pencil-square-o fa guard_signed" style="color:lightgrey" aria-hidden="true" title="' + gt('This Email was signed but unable to verify.') + '"/>');
            }
            if (_.device('small')) {
                signedicon.css('margin-right', '10px');
            }
            this.append(signedicon);
        }
        checkIntegrity(baton.data);
    };

    function checkIntegrity (data) {
        if (data && data.headers) {
            if (data.headers['X-Guard-Failed-Integrity']) {
                require(['io.ox/core/notifications'], function (notify) {
                    notify.yell('warning', gt('This email is missing one or more security checks.  HTML markup was removed for your security.'));
                });
            }
        }
    }

    // Detail View Extensions
    ext.point('io.ox/mail/detail/body').extend({
        id:'guardCheck',
        index: 1,
        draw: function (baton) {
            if (baton.data.PGP) {  // If drawing a PGP email, and already decoded
                // Check if this is a reload, resulting in the attachments changing
                if (baton.data.attachments[0] && !baton.data.attachments[0].pgpFormat) {
                    baton.attachments = baton.data.decodedAttachments;
                    baton.data.attachments = baton.data.decodedAttachments;
                }
            }
        }
    });

    ext.point('io.ox/mail/detail/attachments').extend({
        id: 'ogCheck',
        index: 1,
        draw: function (baton) {
            if (baton.data.security_info && baton.data.security_info.encrypted) baton.stopPropagation();
        }
    });

    ext.point('io.ox/mail/detail/body').extend({
        id: 'ogPGPCheck',
        index: 1,
        draw: function (baton) {
            if (this[0] !== undefined) {
                var location;
                if (this[0].host !== undefined) {  // Chrome
                    location = $(this[0].host);
                } else { // Others
                    location = this[0];
                }
                checker.checkPGP(baton, location);
            }
            if (baton.data.security_info && baton.data.security_info.encrypted) {
                baton.stopPropagation();
            }
        }
    });

    ext.point('io.ox/mail/mobile/detail/body').extend({
        index: 230,
        id: 'ogPGPCheckMobile',
        draw: function (baton) {
            var location = this[0];
            checker.checkPGP.call(this, baton, location);
            baton.view.listenTo(baton.view.model, 'change:attachments', function () {
                checker.checkPGP(baton, location);
                if (baton.view.$el === null) return;
                if (!_.device('ios')) {
                    window.setTimeout(function () {
                        baton.view.$el.find('#oxgrpass').focus();
                    }, 50);
                }

            });
            if (baton.data.security_info && baton.data.security_info.encrypted) {
                baton.stopPropagation();
            }
        }
    });

    // Add lock to header
    ext.point('io.ox/mail/detail/header').extend({
        index: 230,
        id: 'pgpMailLock',
        draw: function (baton) {
            if (baton.data.decrypted) {
                // Add lock symbol to header
                var lockdiv = $('<div style="float:left; padding-right:5px;" id="lockdiv" class="oxguard_lock" ></div>');
                var lockstack = '<span class="icon-stack" title="' + gt('This Email was sent using PGP encryption. It has been decrypted with your private key') + '"><i class="icon-lock icon-large icon-stack-base fa fa-lock -fa-large oxguard_lock" />' +
                    '<i class ="icon-star-half-empty" style="color:black; margin-top:5px;"/></span>';
                lockdiv.append($(lockstack));
                if (baton.data.PGPResults && baton.data.PGPresults.verified === true) {
                    lockdiv.append('<i class="icon-edit" title="' + gt('This Email was signed by the sender and verified.') + '"/>');
                }
                $(this).prepend(lockdiv);
            }
        }
    });

    ext.point('io.ox/mail/view-options').extend({
        id: 'guardThread',
        index: 1,
        draw: function (baton) {
            if (capabilities.has('guest')) {
                baton.app.settings.set('threadSupport', false);
            }
        }
    });

    // Methods for actions
    // Check if attachment filename ends with .asc
    function isASC (e) {
        try {
            if (e.context instanceof Array) {
                for (var i = 0; i < e.context.length; i++) {
                    if (e.context[i].filename.indexOf('.asc') > 0) {
                        // Do not return positive for signatures.  Handled differently
                        if (e.context[i].content_type.indexOf('sig') > 0) return false;
                        return true;
                    }
                }
            }
            var filename = e.context.filename;
            if (filename === undefined) return false;
            if (e.context.content_type.indexOf('sig') > 0) return false;
            return (filename.indexOf('.asc') > 0);
        } catch (d) {
            console.log(d);
            return false;
        }
    }

    // test a single file if it is a key
    function testKey (file) {
        var test = {
                collection: null,
                context: file
            };
        return (isKey (test));
    }

    function isKey (e) {
        if (isASC(e)) {
            if (e.collection === null || e.collection.has('one')) { // only return true for individual files
                var type;
                var name;
                if (e.context instanceof Array) {
                    type = e.context[0].content_type;
                    name = e.context[0].filename;
                } else {
                    type = e.context.content_type;
                    name = e.context.filename;
                }
                if (type !== undefined) {
                    if (type.indexOf('application/pgp-keys') > -1) return true;
                }
                if (name === 'public.asc' || name === 'publickey.asc') return true;
                return (/[0x]*[0-9A-F]{8}\.asc/i).test(name); // Regex for 8 hex.asc
            }
        }
        return false ;
    }

    ext.point('io.ox/mail/detail/attachments').extend({
        id: 'Import',
        draw: function (e) {
            if (capabilities.has('guest')) return;
            var keyfound = false;
            e.attachments.forEach(function (a) {
                if (testKey(a)) {
                    keyfound = true;
                }
            });
            if (keyfound) {
                var keydiv = $('<div class="importKey">');
                var notice = $('<span>' + gt('PGP Public Key Found.  Click to Import') + '</span>');
                notice.click(function () {
                    doImportKey(e.attachments);
                });
                this.append(keydiv.append(notice));
            }
        }
    });

    new links.Action('io.ox/mail/actions/save-encrypted-attachment', {
        capabilities: 'infostore guard-drive',
        requires: function (e) {
            if (!e.collection.has('some')) {
                return false;
            }
            return (util.isDecrypted(e.baton.data));
        },
        multiple: function (list) {
            require(['oxguard/auth', 'io.ox/mail/actions/attachmentSave'], function (auth_core, action) {
                auth_core.authorize(list, gt('Please re-enter your %s password.', window.oxguarddata.productName), true)
                .done(function () {
                    for (var i = 0; i < list.length; i++) {
                        list[i].reEncrypt = true;
                    }
                    action.multiple (list);
                });
            });
        }
    });

    ext.point('io.ox/mail/attachment/links').extend(
        new links.Link({
            id: 'saveEncrypted',
            index: 550,
            mobile: 'high',
            // #. %1$s is usually "Drive" (product name; might be
            // customized)
            label: gt('Save encrypted to %1$s', gt.pgettext('app',
                    'Drive')),
            ref: 'io.ox/mail/actions/save-encrypted-attachment'
        }));

    new links.Action('io.ox/mail/actions/PGPPublicKey', {
        id: 'PGPKey',
        requires: function (e) {
            return isKey(e);
        },
        multiple: function (list) {
            metrics.track('import-pgp-public-key');
            doImportKey (list);
        }
    });

    function doImportKey (list) {
        if (window.oxguarddata.passcode === 'No Key') {
            require(['oxguard/core/createKeys'], function (keys) {
                metrics.track('create-keys-to-import-public');
                keys.createKeysWizard()
                .done(function () {
                    loadPublicKey(list);
                });
            });
        } else {
            loadPublicKey(list);
        }
    }

    /// Sent folder extensions

    ext.point('io.ox/mail/links/inline').extend(new links.Link({
        index: 101,
        prio: 'lo',
        id: 'pinlink',
        label: gt('Check assigned PIN'),
        ref: 'io.ox/mail/actions/pin',
        mobile: 'lo'
    }));

    var unified_sent = '';

    new links.Action('io.ox/mail/actions/pin', {
        id: 'statusaction',
        requires: function (e) {
            if (!(_.contains(account.getFoldersByType('sent'), e.context.folder_id)) && (e.context.folder_id !== unified_sent)) return (false);
            try {
                if (e.baton.data.headers === undefined) return (false);
                return (e.baton.data.headers['X-OxGuard-PIN'] !== undefined);
            } catch (ex) {
                console.log(ex);
                return (false);
            }
        },
        action: function (baton) {
            pin(baton);
        }
    });

    function pin (baton) {
        require (['io.ox/core/tk/dialogs'], function (dialogs) {
            var dialog = new dialogs.ModalDialog({ width: 300, center: true, enter: 'ok' });
            dialog.header($('<h4>').text(gt('PIN')));
            var pin = baton.data.headers['X-OxGuard-PIN'];
            var pintext = $('<h2>').append(pin);
            dialog.getBody().append(pintext);
            dialog.addPrimaryButton('ok', gt('OK'), 'ok').show();
        });
    }

    // We need to update the extensions that the email has security json
    ext.point('io.ox/mail/detail/attachments').extend({
        index: 1,
        id: 'guardDecrypted',
        draw: function (baton) {
            dupSecurity(baton);
        }
    });

    ext.point('io.ox/mail/mobile/detail/attachments').extend({
        index: 1,
        id: 'guardDecrypted',
        draw: function (baton) {
            dupSecurity(baton);
        }
    });

    ext.point('io.ox/mail/externalImages').extend({
        index: 1,
        id: 'checkAuth',
        perform: function (baton) {
            var def = $.Deferred();
            if (baton.data.security && baton.data.security.decrypted) {
                require(['oxguard/auth'], function (auth_core) {
                    auth_core.authorize(baton, gt('Please re-enter your %s password.', window.oxguarddata.productName), true)
                    .done(function () {
                        def.resolve();
                    })
                    .fail(def.reject);
                });
            } else {
                def.resolve();
            }
            return def;
        }
    });

    function dupSecurity(baton) {
        if (baton.data.security && baton.data.attachments.length > 0) {
            for (var i = 0; i < baton.data.attachments.length; i++) {
                baton.data.attachments[i].security = baton.data.security;
            }
        }
    }

    // Function to do authentication validation before calling an extension.
    // Checks if valid.  If not, prompts for password, calls extension again once auth
    var validated = false;
    function doAuthCheck(extName, baton, array, storeAuth) {
        if (validated) {  // If already validated, return so that extension propagates
            validated = false;
            return;
        }
        baton.stopPropagation(); // Stop this extension and check authorization

        // Local decryption check.  If we decrypted locally, we will have decrypted sessionKey
        if (baton.data &&
                (baton.data[0] && baton.data[0].security && baton.data[0].security.sessionKey) ||
                (baton.data.security && baton.data.security.sessionKey)) {
            require (['pgp_local/sessionHandler'], function (handler) {
                var key = _.isArray(baton.data) ? baton.data[0].security.sessionKey : baton.data.security.sessionKey;
                handler.saveDecryptedSession(key)
                .done(function () {
                    validated = true;
                    baton.resumePropagation();
                    actions.invoke(extName, array, baton);  // OK, authorized, call the extension again
                })
                .fail(function () {
                    window.setTimeout(function () {
                        baton.resumePropagation();
                    }, 500);
                });
            });
            return;
        }

        // Not local, so authorize against backend
        require(['oxguard/auth', 'io.ox/core/extPatterns/actions'], function (auth_core, actions) {
            auth_core.authorize(baton, gt('Please re-enter your %s password.', window.oxguarddata.productName), true)
            .done(function (authData) {
                if (baton.data) {
                    if (_.isArray(baton.data)) {
                        baton.data.forEach(function (d) {
                            d.auth = authData;
                        });
                    } else {
                        baton.data.auth = authData;
                    }
                }
                baton.resumePropagation();
                validated = true;
                if (storeAuth) {
                    if (baton.data) {
                        baton.data.security = _.extend({ authentication: authData }, baton.data.security);
                    }
                }
                actions.invoke(extName, array, baton);  // OK, authorized, call the extension again
            })
            .fail(function () {
                baton.stopPropagation();
                window.setTimeout(function () {
                    baton.resumePropagation();
                }, 500);
            });
        });
    }

    registerPasswordCheck('io.ox/mail/actions/reply', true);
    registerPasswordCheck('io.ox/mail/actions/reply-all', true);
    registerPasswordCheck('io.ox/mail/actions/forward', true);
    registerPasswordCheck('io.ox/mail/actions/edit', true);
    registerPasswordCheck('io.ox/mail/actions/print');
    registerPasswordCheck('io.ox/mail/actions/view-attachment');
    registerPasswordCheck('io.ox/mail/actions/open-attachment');
    registerPasswordCheck('io.ox/mail/actions/download-attachment');
    registerPasswordCheck('io.ox/mail/actions/save-attachment');
    registerPasswordCheck('io.ox/mail/office/text-edit-asnew');
    registerPasswordCheck('io.ox/mail/office/spreadsheet-edit-asnew');
    registerPasswordCheck('io.ox/mail/actions/vcard');
    registerPasswordCheck('io.ox/mail/actions/ical');

    // Function to check actions for Guard emails, and if found, proper authentication
    function registerPasswordCheck (extName, storeAuth) {
        ext.point(extName).extend({
            id: 'guardPasswordCheck',
            index: 1,
            requires: function () {
                return false;  // Always false, depend on other requires to return true
            },
            multiple: function (array, baton) {
                if (storeAuth) {
                    if (util.isEncryptedMail(baton.data)) {
                        baton.data.security = {
                            decrypted: true
                        };
                        doAuthCheck(extName, baton, array, storeAuth);
                        return;
                    }
                }
                if (!util.isDecrypted(baton.data)) return;
                doAuthCheck(extName, baton, array, storeAuth);
            }
        });
    }

    /**
     * Extension to check if allowed to forward decrypted/encrypted email
     * Must have Guard-mail capability
     */
    ext.point('io.ox/mail/actions/forward').extend ({
        id: 'guardLimitedReply',
        index: 10,
        requires: function (e) {
            if (e.baton) {
                if (util.isDecrypted(e.baton.data) || util.isEncryptedMail(e.baton.data)) {
                    if (!capabilities.has('guard-mail')) {
                        e.stopPropagation();
                        return false;
                    }
                }
            }
        }
    });

    ////////// End extensions
    /// Functions

    // Save public key from attachment
    function loadPublicKey (list) {
        _(list).each(function (data) {
            if (testKey(data)) { // make sure key
                var params = '&emailid=' + data.parent.id +
                '&attach=' + data.id +
                '&userid=' + ox.user_id +
                '&cid=' + ox.context_id +
                '&folder=' + data.parent.folder_id;
                if (data.security && data.security.decrypted) { //If was encrypted
                    require(['oxguard/auth'], function (auth) {
                        auth.authorize(list).then(function (auth) {
                            params = params + '&auth=' + encodeURIComponent(auth) +
                            '&inline=' + (data.security.pgpInline ? 'true' : 'false') +
                            '&filename=' + encodeURIComponent(data.filename);
                            doLoadPublicKey (params);
                        });
                    });
                } else {
                    doLoadPublicKey (params);
                }

            }
        });
    }

    function doLoadPublicKey (params) {
        var link = ox.apiRoot + '/oxguard/pgpmail/?action=savepublicattach' + params;
        http.get(link, '')
        .done(function (data) {
            require(['io.ox/core/notifications'], function (notify) {
                var keys = JSON.parse(data);
                var added = gt('Added keys: \r\n');
                for (var i = 0; i < keys.length; i++) {
                    added = added + gt('Key ID: ') + keys[i].Key + ' ' + keys[i].ids + '\r\n';
                }
                notify.yell('success', added.replace(/&lt;/g,'<').replace(/&gt;/g,'>'));
            });
        })
        .fail(function (e) {
            console.log(e);
            require(['io.ox/core/notifications'], function (notify) {
                if (e.status === 503) {
                    notify.yell('error', gt('Service Unavailable'));
                } else
                    notify.yell('error', e.responseText);
            });
        });
    }
});
