/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Author Greg Hill <greg.hill@open-xchange.com>
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/pgp/keyDetails', [
    'gettext!oxguard',
    'oxguard/core/og_http',
    'oxguard/oxguard_core',
    'less!oxguard/style',
    'less!oxguard/pgp/style'
], function (gt, http, core) {

    var deleting = false;

    function hideWait () {
        $('#og_wait').hide();
    }

    var keyRingsCache = {};

    // Confirm deletion of item
    function deleteConfirm (e) {
        e.preventDefault();
        require (['oxguard/core/confirmView'], function (confirm) {
            confirm.open(gt('Please confirm you want to delete this key.'))
            .done(function () {
                var id = $(e.target).parent().attr('value');
                deleteShare (id);
            });
        });
    }

    // Delete share
    function deleteShare (ids) {
        if (deleting) return;
        deleting = true;
        var params = '&keyids=' + encodeURI(ids) +
            '&session=' + ox.session;
        core.metrics('settings', 'delete-public-key');
        $.ajax({
            url: ox.apiRoot + '/oxguard/keys?action=deleteExternalPublicKey' + params,
            type: 'DELETE',
            success: function (data) {
                core.checkJsonOK(data);
                $('.refreshkeys').click();
                deleting = false;
            }
        })
        .fail (function (data) {
            console.log(data);
            alert ('fail');
            deleting = false;
        });
    }

    // Guard sends seperate keys for each ID, consolidate by keyid
    /*
    function consolidate (data) {
        var newData = {};
        for (var i = 0; i < data.length; i++) {
            var id = data[i].ids.replace(/\s/g,'');
            if (newData[id] === undefined) {
                newData[id] = data[i];
            } else {
                if (newData[id].shared && !data[i].shared) {  // If this is owned and not shared, then should be used
                    newData[id] = data[i];
                }
            }
        }
        return newData;
    }
    */

    function getUserIds (publicRing) {
        var ids = '';
        if (publicRing.keys && publicRing.keys.length > 0) {
            for (var i in publicRing.keys) {
                var key = publicRing.keys[i];
                if (key.userIds && key.userIds.length > 0) {
                    for (var k in key.userIds) {
                        if (key.userIds[k] && key.userIds[k].length > 1) {
                            ids += key.userIds[k] + ' ';
                        }
                    }
                }
            }
        }
        return ids;
    }

    // Switch depending on if specified contact
    function updateKeyTable(id, folder_id, contact_id) {
        if (folder_id && contact_id) {
            return getContactTable(id, folder_id, contact_id);
        } else {
            return doUpdateKeyTable(id);
        }
    }

    // Draws key table with list of keys
    // List is filter for address book
    function doUpdateKeyTable(id) {
        var deferred = $.Deferred();
        // Send the request to the oxguard server for the encrypted content key to decrypt
        http.get(ox.apiRoot + '/oxguard/keys?action=getExternalPublicKeys', '')
        .done(function (result) {
            if (core.checkJsonOK(result)) {
                var data = result.data.externalPublicKeyRings;
                $('#keylisttable' + id).replaceWith(getTable(data, id));
                deferred.resolve('OK');
            }
            hideWait();

        });
        return (deferred);
    }

    function getContactTable(id, folder_id, contact_id) {
        var deferred = $.Deferred();
        var params = '&contactFolder=' + folder_id + '&contactId=' + contact_id;
        http.get(ox.apiRoot + '/oxguard/keys?action=getContactKeys', params)
        .done(function (result) {
            if (core.checkJsonOK(result)) {
                var data = result.data.externalPublicKeyRings;
                $('#keylisttable' + id).replaceWith(getTable(data, id));
                deferred.resolve('OK');
            }
            hideWait();

        });
        return (deferred);
    }

    function getFilename (email) {
        var filename = '';
        if (email.indexOf('&lt') > 0) {  // If has name (i.e. text before the email address, use the name
            filename = email.substring(0, email.indexOf('&lt')).trim() + '.asc';
        }
        // Return if name could be proper filename.  Otherwise just return public.asc
        //#.  public used for filename for a public key.  I.E. the filename will be public.asc
        return /^[a-z0-9_ .@()-]+$/im.test(filename) ? filename : gt('public') + '.asc';
    }
    function getTable (keyRings, tableId) {
        var showDelete = false;
        var newtable = $('<table class="keytable" id="keylisttable' + tableId + '">');
        //#. Table headers for list of keys, option for Details and to delete
        var headers = $('<th>' + gt('Email') + '</th><th>' + gt('Key Ids') + '</th><th>' + gt('Details') + '</th><th class="keyDelete">' + gt('Delete') + '</th><th>' + gt('Download') + '</th>');
        keyRingsCache = {};
        if (keyRings && keyRings.length > 0) {
            newtable.append(headers);
            for (var i in keyRings) {
                var tr = $('<tr>');
                if (keyRings[i].guardKey === true || keyRings[i].owned === false) tr.addClass('oxguard_key_share');
                if (keyRings[i].expired === true) tr.css('color', 'red');
                var email = getUserIds(keyRings[i].publicRing);
                email = email.replace(/>/gi,'&gt;').replace(/</gi,'&lt;');
                var td1 = $('<td class="oxguard_pubkeylist" title="' + email + '">');
                td1.append(email);
                tr.append(td1);
                var td2 = $('<td class="oxguard_pubkeylist" title="' + keyRings[i].ids + '">');
                td2.append(keyRings[i].ids);
                tr.append(td2);
                var td4 = $('<td>');
                var search = $('<a class="pgpdetail" ids="' + keyRings[i].ids + '" notowned="' + (((keyRings[i].guardKey === true) || (keyRings[i].owned === false)) ? 'true' : 'false') + '" href="#" title = "' + gt('Details') + '"><icon class="fa fa-search"/></a>');
                search.on('click', detail);
                td4.append(search);
                var td5 = $('<td class="keyDelete">');
                if ((keyRings[i].guardKey !== true) && (keyRings[i].owned === true)) {
                    var del = $('<a href="#" value="' + keyRings[i].ids + '" tableid="' + tableId + '" title = "' + gt('Delete') + '"><icon class="fa fa-trash"/></a>')
                    .click (deleteConfirm);
                    td5.append(del);
                    showDelete = true;
                }
                var td6 = $('<td class="keyDownload">');
                var uri = 'data:text/plain;charset=utf-8,' + encodeURIComponent(keyRings[i].publicRing.ring);
                var download = $('<a href="#" ids="' + keyRings[i].ids + '" title = "' + gt('Download') + '"><icon class="fa fa-download"/></a>')
                .attr('href', uri).attr('download', getFilename(email));
                td6.append(download);
                tr.append(td4).append(td5).append(td6);
                newtable.append(tr);
                keyRingsCache[keyRings[i].ids] = keyRings[i];
            }
            if (!showDelete) {  // If only system keys, then hide the delete column
                newtable.find('.keyDelete').empty();
            }
            return newtable;
        } else {
            return ('<table id="keylisttable' + tableId + '"><tr><td>' + gt('No Keys Found') + '</td></tr></table>');
        }
    }

    // Basic list of public key information
    // Shows keyDetail function output, plus adds buttons for sharing, etc in dialog
    // Shown with the inspect icon in pgp public key list
    function detail (e) {
        var id = $(this).attr('ids');
        var guard = $(this).attr('notowned');
        var keyData = keyRingsCache[id];
        if (keyData) {
            var keydetail = keyDetail(keyData.publicRing.keys);
            ox.load(['oxguard/pgp/detailView']).then(function (view) {
                view.open(keydetail, keyData, guard);
            });
        }
        e.preventDefault();
    }

    //  Shows details of keys in div
    //  Can be called for local keys, or for pgp public key list
    //  Minimal triggers display of details such as master, signing, etc or not
    function keyDetail (keys, minimal) {
        var keyids = '';
        var keylong = ''; // This will be the key that has the ids (usually master)
        var images = null;
        var resultdiv = $('<div>');
        for (var i in keys) {
            var key = keys[i];
            //keyids = keyids + data[i].Key + ' ';  // List of key ids for changing share, inline (linked)
            if (i > 0) resultdiv.append('<hr>');
            var detaildiv = $('<div id="keyDetail"' + (minimal ? ' >' : ' style="padding-left:20px;">'));
            //#. if key is "expired"
            if (key.expired) {
                detaildiv.append('<b>' + gt('Expired') +'</b><br>');
                detaildiv.css('color', 'red');
            }
            if (key.revoked) detaildiv.append('<b style="color:red">' + gt('Revoked') +'</b><br>');
            detaildiv.append((key.privateKey ? ('<b>' + gt('Private Key Available') + '</b>') : gt('Public Key')) + '<br>');
            //#. Fingeprint of the key
            detaildiv.append(gt('Fingerprint:') + ' ' + formatFingerprint(key.fingerPrint) + '<br>');
            detaildiv.append(gt('Created:') + ' ' + new Date(key.creationTime) + '<br>');
            if (!minimal) {
                //#. Key is a Master key
                detaildiv.append(gt('Master:') + ' ' + (key.masterKey ? gt('True') : gt('False')) + ',  ');
                //#. Key is an encryption key
                detaildiv.append(gt('Encryption:') + ' ' + (key.usedForEncryption ? gt('True') : gt('False')));
                if (key.usedForSigning) {
                    detaildiv.append(', ' + gt('Sign:') + ' ' + gt('True') +'<br>');
                } else {
                    detaildiv.append('<br>');
                }
                var ids = '';
                for (var j in key.userIds) {
                    if (j > 0) ids = ids + ', ';
                    ids = ids + key.userIds[j].replace(/>/gi,'&gt;').replace(/</gi,'&lt;');
                }
                if (ids !== '') {
                    detaildiv.append(gt('IDs: ') + ids + '<br>');
                    keylong = key.id;  // The ID of the key with IDs for signature verification
                }
                if (key.images !== undefined) {
                    if (key.images.length > 0) {
                        images = key.images;
                        var image = $('<img alt="Embedded Image" id="imageID" src="' +  key.images[0].imageData + '" style="float:right;max-height:50px;max-width:50px;"/>');
                        detaildiv.append(image);
                    }
                }
                if (key.validSeconds) {
                    detaildiv.append(gt('Expires: %1$s', new Date(key.creationTime + (key.validSeconds * 1000))) + '<br>');
                }
            }
            resultdiv.append(detaildiv);
        }
        return {
            div: resultdiv,
            keyids: keyids,
            keylong: keylong,
            images: images
        };
    }

    function formatFingerprint (fp) {
        if (fp.length < 28) return (fp);
        if (fp.indexOf(' ') < 0) { // if not spaced
            var newFingerprint = '';
            for (var i = 0; i < fp.length; i = i + 4) {
                newFingerprint += fp.substring(i, i + 4) + ' ';
            }
            fp = newFingerprint;
        }
        var line1 = fp.substring (0,24);
        var line2 = fp.substring (25, fp.length-10) + '<b>' + fp.substring(fp.length-10) + '</b>';
        return ('<span style="float:right;">' + line1 + '</span><br/><span style="float:right">' + line2 + '</span>');
    }

    function update(id, folder_id, contact_id) {
        $('.refreshkeys').addClass('fa-spin');
        updateKeyTable(id, folder_id, contact_id).done(function () {
            $('.refreshkeys').removeClass('fa-spin');
        });
    }

    // Generate list of public keys
    function listPublic (id, folder_id, contact_id) {
        var pubdiv = $('<div class="publicKeyDiv"/>');
        var tableHeader = $('<legend>');
        if (id == 'settings') tableHeader = $('<b>');
        tableHeader.append(gt('PGP Public Key List'));
        var refresh = $('<a href="#" style="margin-left:10px;" id="refreshkeys" title = "' + gt('Refresh') + '"><icon class="fa fa-refresh fa-spin refreshkeys"></a>')
            .click(function (e) {
                $(this).addClass('fa-spin');
                updateKeyTable(id, folder_id, contact_id).done(function () {
                    refresh.removeClass('fa-spin');
                });
                e.preventDefault();
            });
        var fileinput = $('<input type="file" name="publicKey" style="display:none;" multiple>');
        var addKey = $('<a href="#" style="margin-left:10px;" title = "' + gt('Add Public Key') + '"><icon class="fa fa-plus" id="addpgppublickeys"></a>')
        .click(function (e) {
            fileinput.unbind('change');
            fileinput.on('change', function () {
                var files = this.files;
                if (files.length > 0) {
                    core.metrics('settings', 'upload-public-key');
                    require(['oxguard/pgp/uploadkeys'], function (uploader) {
                        uploader.uploadExternalKey(files)
                        .done(function () {
                            updateKeyTable(id, folder_id, contact_id);
                        })
                        .always(function () {
                            fileinput.val('');
                        });
                    });
                }
            });
            fileinput.click();
            e.preventDefault();
        });
        tableHeader.append(refresh).append(fileinput).append(addKey);
        var tablediv = $('<div class="keytable">');
        var table = $('<table id="keylisttable' + id + '">');
        pubdiv.append(tableHeader).append(tablediv.append(table));
        update(id, folder_id, contact_id);
        return (pubdiv);
    }

    return {
        listPublic: listPublic,
        keyDetail: keyDetail
    };
});
