/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

define('oxguard/pgp/detailView', [
    'io.ox/backbone/views',
    'io.ox/core/extensions',
    'io.ox/backbone/views/modal',
    'io.ox/core/settings/util',
    'io.ox/core/yell',
    'oxguard/oxguard_core',
    'gettext!oxguard',
    'less!oxguard/pgp/style'
], function (views, ext, ModalView, util, yell, core, gt) {

    'use strict';

    var POINT = 'oxguard/pgp/detailview',
    INDEX = 0;

    function open (table, keyData) {
        return openModalDialog(table, keyData);
    }

    function openModalDialog(table, keyData) {

        var dialog = new ModalView({
            async: true,
            point: POINT,
            title: gt('Public Keys Detail'),
            id: 'pkeyDetail',
            width: 640,
            model: new Backbone.Model({ 'keyData': keyData, 'detail': table })
        })
        .inject({
            showSignatures: function () {
                // Pull list of signatures
                ox.load(['oxguard/pgp/signatureView']).done(function (view) {
                    view.open(table.keylong);
                });
            }
        })
        .addButton({ label: gt('Done'), action: 'done' })
        .on('done', function () {
            $('.refreshkeys').click();
            this.close();
        })
        .on('signatures', function () {
            this.showSignatures();
            this.idle();
        });
        if (!keyData.localKeys) {
            dialog.addAlternativeButton({ label: gt('Signatures'), action: 'signatures' });
        }
        dialog.open();
        return dialog;
    }

    ext.point(POINT).extend(
            // Draw Div with key details
            {
                index: INDEX += 100,
                id: 'detail',
                render: function (baton) {
                    var div = $('<div>');
                    this.$body.append(
                        div.append(baton.model.get('detail').div)
                    );
                }
            },
            // Share / Inline
            {
                index: INDEX += 100,
                id: 'share',
                render: function (baton) {
                    var data = baton.model.get('keyData');
                    if (data && !data.guardKey && !data.shared && !data.localKeys) { // we don't display share options for Guard created keys
                        var optionsdiv = $('<div class="keyoptions">');
                        var sharingtext = $('<label>' + gt('Share Keys') + '</label>');
                        var sharing = $('<input type="checkbox" value="' + data.ids + '" style="margin-right:15px;">');
                        sharing.prop('checked', data.shareLevel > 0)
                        .on('click', changeShare);
                        sharingtext.click(function () {
                            sharing.click();
                        });
                        var inlinetext = $('<label>' + gt('Use PGP Inline') + '</label>');
                        var inline = $('<input type="checkbox" value="' + data.ids + '" style="margin-right:15px; margin-left:40px;">')
                        .on('click', changeInline);
                        inline.prop('checked', data.inline);
                        inlinetext.click(function () {
                            inline.click();
                        });
                        this.$body.append(
                            optionsdiv.append(sharing).append(sharingtext).append(inline).append(inlinetext)
                        );
                    }
                }
            }
     );

    // Change the status of a key to sharing or not
    function changeShare (e) {
        var params = '&ids=' + encodeURI(e.target.value) + '&share=' +
            ($(e.target).is(':checked') ? 'true' : 'false') +
            '&session=' + ox.session;
        wait();
        $.ajax({
            url: ox.apiRoot + '/oxguard/keys?action=shareExternalPublicKey' + params,
            type: 'PUT',
            success: function (data) {
                core.checkJsonOK(data);
                hideWait();
            }
        })
        .fail (function (data) {
            hideWait();
            console.log(data);
            yell('error', data.responseText);
        });
    }

    // Change the status of a key to sharing or not
    function changeInline (e) {
        var params = '&ids=' + encodeURI(e.target.value) + '&inline=' +
            ($(e.target).is(':checked') ? 'true' : 'false') +
            '&session=' + ox.session;
        wait();
        $.ajax({
            url: ox.apiRoot + '/oxguard/keys?action=inlineExternalPublicKey' + params,
            type: 'PUT',
            success: function (data) {
                core.checkJsonOK(data);
                hideWait();
            }
        })
        .fail (function (data) {
            console.log(data);
            hideWait();
            yell('error', data.responseText);
        });
    }

    function wait () {
        $('#og_wait').show();
    }

    function hideWait () {
        $('#og_wait').hide();
    }

    return {
        open: open
    };

});
