/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/oxguard_core', [
    'settings!io.ox/mail',
    'settings!oxguard',
    'oxguard/core/og_http',
    'io.ox/core/api/user',
    'gettext!oxguard'
], function (settings, guardSettings, og_http, userAPI, gt) {
    'use strict';

    var badCount = 0;

    //// Utils

    function validateEmail(email) {
        var re = /^(([^<>()[\]\\.,;:\s@\"]+(\.[^<>()[\]\\.,;:\s@\"]+)*)|(\".+\"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
        return re.test(email);
    }

    // Unified inbox combines folder and email ID, returns email id

    function metrics (app, action) {
        require(['io.ox/metrics/main'], function (metrics) {
            if (!metrics.isEnabled()) return;
            metrics.trackEvent({
                app: app,
                target: 'guard',
                type: 'click',
                action: action
            });
        });
    }

    function sanitize (data) {
        var regex = new RegExp('<(?:!--(?:(?:-*[^->])*--+|-?)|script\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</script\\s*|style\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</style\\s*|/?[a-z](?:[^"\'>]|"[^"]*"|\'[^\']*\')*)>', 'gi');
        return (data.replace(regex, ''));
    }

    //// Failure Handling

    function handleJsonError (error, errorResponse, baton) {
        errorResponse.errorMessage = gt('Error');
        errorResponse.retry = false;
        if (error.code !== undefined) {
            switch (error.code) {
                case 'GRD-PGP-0005':
                    if (window.oxguarddata.passcode && (window.oxguarddata.passcode.length > 20)) {  // If bad password based on auth, then clear auth and redo
                        window.oxguarddata.passcode = null;
                        if (baton) baton.view.redraw();
                        return;
                    }
                    errorResponse.errorMessage = gt('Unable to decrypt Email, incorrect password.');
                    require (['io.ox/core/notifications'], function (notifications) {
                        notifications.yell('error', gt('Bad password'));
                    });
                    badCount++;
                    if (badCount > 2 && window.oxguarddata.recoveryAvail) {
                        errorResponse.errorMessage += ('<br/>' + gt('To reset or recover your password, click %s', '<a id="guardsettings" href="#">' + gt('Settings') + '</a>'));
                    }
                    errorResponse.retry = true;
                    break;
                default:
                    if (error.error !== undefined) {
                        errorResponse.errorMessage = error.error;
                    }
            }

        }
    }

    function showError (errorJson) {
        var errorResp = {};
        try {
            handleJsonError (errorJson, errorResp);
        } catch (e) {
            errorResp.errorMessage = gt('Unspecified error');
        }
        require (['io.ox/core/notifications'], function (notifications) {
            notifications.yell('error', errorResp.errorMessage);
        });

    }

    function checkJsonOK (json) {
        if (json && json.error) {
            showError(json);
            return false;
        }
        return true;
    }

    /// Authentication Section

    // Password prompt
    function getpassword (message, timeOption) {
        var def = $.Deferred();
        require(['oxguard/core/passwordPrompt'], function (prompt) {
            prompt.open(message, timeOption)
            .done(function (data) {
                def.resolve(data);
            })
            .fail(function (e) {
                def.reject(e);
            });
        });
        return (def);
    }

    // Save existing authentication token to OX Session
    function saveAuthToSession (auth, time) {
        var def = $.Deferred();
        var url = ox.apiRoot + '/guard-json?action=auth-token';
        var params = '';
        var authToken = {
            auth: auth,
            minutesValid: (time === undefined ? -1 : time)
        };
        if (auth === null) authToken = {};
        if (time !== undefined) {
            authToken.minutesValid = time;
        }
        og_http.put (url, params, authToken)
        .done (function (data) {
            def.resolve(data.auth);
        })
        .fail(function () {
            def.reject();
        });
        return def;
    }

    // Remove authentication from session
    function removeAuth() {
        window.oxguarddata.passcode = null;
        return saveAuthToSession('');
    }

    // Save a password in both auth code and token
    function savePassword (password, duration) {
        var def = $.Deferred();
        if (duration > 9999) duration = 0;  // 0 for session
        authAndSave (password, duration)
        .done(function (data) {
            def.resolve(data.data);
            storeAuth (data.data.auth, duration);
        })
        .fail(function (e) {
            def.reject(e);
            require (['io.ox/core/notifications'], function (notify) {
                switch (e.code) {
                case 'GRD-MW-0003':
                    notify.yell('error', gt('Bad password'));
                    break;
                case 'GRD-MW-0005':
                    notify.yell('error', gt('Account locked out due to bad attempts.  Please try again later.'));
                    break;
                default:
                    notify.yell('error', gt('Error checking password') + '\r\n' + e.error);
                    break;
                }
                console.error(e);
            });
        });
        return def;
    }

    // Store the auth locally for a time
    function storeAuth (auth, duration) {
        if (duration >= 0) {
            window.oxguarddata.passcode = auth;
            var time = duration * 1000 * 60;
            if (time > 0) {  // 0 is indefinite
                window.setTimeout(function () {
                    window.oxguarddata.passcode = null;
                }, time);
            }
        }
    }

    // Check if authentication token in session and valid
    function checkAuth () {
        var def = $.Deferred();
        var url = ox.apiRoot + '/guard-json?action=auth-token';
        og_http.get (url, '')
        .done (function (data) {
            if (data.data === undefined) {
                def.reject('none');
                return;
            }
            data = data.data;
            if (data.minutesValid !== undefined) {
                if (data.minutesValid >= -1) { // -1 is single use
                    if (data.minutesValid >= 0) { // If more than single use, store locally
                        storeAuth (data.auth, data.minutesValid);
                    }
                    def.resolve(data); // OK only if exists and still time left.
                    return;
                } else {
                    window.oxguarddata.passcode = null;
                }
            }
            def.reject('exp');
        })
        .fail (function (e) {
            console.log(e);
            def.reject('fail');
        });
        return def;
    }

    // Send password for authentication and save it to session
    function authAndSave (password, duration) {
        var def = $.Deferred();
        var userdata = {
            password: password,
            minutesValid: duration
        };
        var url = ox.apiRoot + '/guard-json?action=auth-token';
        var params = '&time=' + new Date().getTime();
        og_http.post(url, params, userdata)
        .done (function (data) {
            if (data.error) {
                def.reject(data);
                return;
            }
            def.resolve(data);
        })
        .fail (function (e) {
            def.reject(e);
        });
        return def;

    }

    // Authenticate against the Guard server, checking for specific key ID, return settings info
    function authorize(userid, password, extra, keyid)
    {
        var def = $.Deferred();
        userAPI.get().then(function (userData) {
            var defaultSendAddress = $.trim(settings.get('defaultSendAddress', ''));
            var userdata = {
                encr_password: password,
                lang: ox.language,
                email: defaultSendAddress ? defaultSendAddress : userData.email1
            };
            var url = ox.apiRoot + '/oxguard/login?action=login';
            var params = '&time=' + new Date().getTime();
            if (keyid !== undefined) params += '&keyid=' + keyid;
            og_http.post (url, params, userdata)
            .done (function (data) {
                def.resolve(data);
            })
            .fail(function () {
                def.reject();
            });
        });
        return def;
    }

    function hasLocalKeys () {
        return (guardSettings.get('localKeys') && guardSettings.get('pgplocal'));
    }

    return {
        auth: authorize,
        validateEmail: validateEmail,
        getPassword: getpassword,
        showError: showError,
        sanitize: sanitize,
        checkJsonOK: checkJsonOK,
        saveAuthToSession: saveAuthToSession,
        checkAuth: checkAuth,
        savePassword: savePassword,
        removeAuth: removeAuth,
        metrics: metrics,
        storeAuth: storeAuth,
        hasLocalKeys: hasLocalKeys
    };
});

