/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */

define('oxguard/mail/keymanager', ['oxguard/core/og_http',
    'gettext!oxguard',
    'oxguard/oxguard_core',
    'settings!io.ox/calendar'],
    function (og_http, gt, core, calSettings) {
    'use strict';

    var blackList;   // We only want to pull the blacklist once

    // Check the list of previously fetched key results to see if the email is there.  If so, return, otherwise query backend
    function checkRecips (email, baton) {
        var def = $.Deferred();
        var gkeys = baton.model.get('gkeys');
        if (gkeys === undefined) { // If undefined, create gkeys now
            gkeys = [];
            baton.model.set('gkeys', gkeys);
            getKeyResult (email, baton)
            .done(function (keyResults) {
                def.resolve(keyResults);
            });
            return def;
        }
        if (gkeys[email] && !gkeys[email].pending) {
            def.resolve(gkeys);
            return def;
        }
        if (gkeys[email] && gkeys[email].pending) {
            return gkeys[email].def;
        }
        getKeyResult (email, baton)
        .done(function (keyResults) {
            def.resolve(keyResults);
        });
        return def;
    }

    // Query guard server for key result
    function getKeyResult (email, baton) {
        var def = $.Deferred();
        var result = {
            email: email,
            result: null,
            pending: true,
            def: def
        };
        var gkeys = baton.model.get('gkeys');
        gkeys[email] = result;
        baton.model.set('gkeys', gkeys);
        if (isBlackListed(email)) {
            require (['io.ox/core/notifications'], function (notifications) {
                notifications.yell('error', gt('Mailing list / blocked email found.  Encrypted emails to this email will fail.  %s', email));
            });
            baton.model.set('encrypt', false);
            def.resolve(updateKeys (email, baton, 'fail'));
            return def;
        }
        var params = '&email=' + encodeURIComponent(email);
        og_http.get(ox.apiRoot + '/oxguard/keys?action=getRecipKey', params)
        .done(function (result) {//Get the key
            try {
                var data;
                if ($.isArray(result)) {  // error responses sent as array
                    data = result[0];
                } else {
                    data = result.data;
                }
                if (core.checkJsonOK(result)) {
                    if (data.preferredFormat === 'PGP/INLINE') { // If needs inline, then set
                        if (baton.model.set('PGPFormat', 'inline'));
                    }
                    var keys = (data.publicRing && data.publicRing.keys) ? data.publicRing.keys : undefined;
                    var encryptionKey;
                    if (keys) {
                        keys.forEach(function (key) {
                            if (key.encryptionKey && !key.expired && !key.revoked) {
                                encryptionKey = key;
                                encryptionKey.keySource = data.keySource;
                                var lang = ox.language.substring(0,2);
                                encryptionKey.Created = (new Date(key.creationTime)).toLocaleDateString(lang, { year:'2-digit', month:'2-digit', day:'2-digit' });
                            }
                        });
                    }
                    if (encryptionKey || data.newKey) {
                        // Expired keys.  Should never occur.  Remove code?
                        if (encryptionKey && encryptionKey.expired === true) {
                            def.resolve(updateKeys (email, baton, 'fail', encryptionKey));
                            require (['io.ox/core/notifications'], function (notifications) {
                                notifications.yell('error', gt('The key for %s has expired.  You will not be able to send an encrypted email to this recipient.', data.email));
                            });
                            return def;
                        }
                        //
                        if (data.guest && data.newKey) {
                            def.resolve(updateKeys (email, baton, 'guest', encryptionKey));
                        } else {
                            def.resolve(updateKeys (email, baton, (data.newKey ? 'new' : 'pgp'), encryptionKey));
                        }
                    } else {
                        def.resolve(updateKeys (email, baton, 'fail', encryptionKey));
                    }
                } else {
                    def.resolve(updateKeys (email, baton, 'fail', data));
                }

            } catch (e) {
                console.log(e);
            }
        })
        .fail (function () {
            def.resolve(updateKeys (email, baton, 'fail'));
        });
        return def;
    }

    // Check if email is in blacklist
    function isBlackListed (email) {
        if (!blackList) {  // Initialize blacklist
            blackList = calSettings.get('participantBlacklist');
            if (blackList) {
                blackList = blackList.toLowerCase();
            } else {
                blackList = '';
            }
        }
        if (blackList && blackList !== '') {
            if (blackList.indexOf(email.toLowerCase()) > -1) {
                return true;
            }
        }

        return false;
    }

    // Update the list of gkeys with the result
    function updateKeys (email, baton, result, keyData) {
        var gkeys = baton.model.get('gkeys');
        gkeys[email] = {
            result: result,
            pending: false,
            key: keyData
        };
        baton.model.set('gkeys', gkeys);
        baton.model.trigger('change:gkeys');
        return gkeys;
    }

    function updateTokens(baton) {
        baton.model.trigger('change:to', baton.model, baton.model.get('to'));
        baton.model.trigger('change:cc', baton.model, baton.model.get('cc'));
        baton.model.trigger('change:bcc', baton.model, baton.model.get('bcc'));
    }

    function checkAll(baton) {
        updateTokens(baton);
    }

    // Delete specific email from gkeys array
    function deleteRecip (email, baton) {
        var gkeys = baton.model.get('gkeys');
        delete gkeys[email];
        baton.model.set('gkeys', gkeys);
    }

    // Check if an email is located in gkeys
    function isPresent (email, recips) {
        for (var i = 0; i < recips.length; i++) {
            if (recips[i][1] === email) return true;
        }
        return false;
    }

    // Clean the list of gkeys, remove tokens that were deleted
    function clean(baton) {
        var to = baton.model.get('to') || [];
        var cc = baton.model.get('cc') || [];
        var bcc = baton.model.get('bcc') || [];
        var total = to.concat(cc).concat(bcc);
        var gkeys = baton.model.get('gkeys') || [];
        var newkeys = [];
        for (var email in gkeys) {
            if (isPresent(email, total)) {
                newkeys[email] = gkeys[email];
            }
        }
        baton.model.set('gkeys', newkeys);
    }

    // Check if Guest present in list
    function isGuest(baton) {
        try {
            clean(baton);
            var keys = baton.model.get('gkeys');
            if (keys === undefined) keys = [];
            for (var email in keys) {
                if (keys[email].result === 'guest') return true;
            }
        } catch (e) {
            console.log(e);
        }

        return false;
    }

    function getGuests(baton) {
        var guests = [];
        try {
            var keys = baton.model.get('gkeys');
            if (keys === undefined) keys = [];
            for (var email in keys) {
                if (keys[email].result === 'guest') guests.push(email);
            }
        } catch (e) {
            console.log(e);
        }
        return (guests);
    }

    return {
        checkRecips: checkRecips,
        checkAll: checkAll,
        clean: clean,
        isGuest: isGuest,
        getGuests: getGuests,
        deleteRecip: deleteRecip
    };

});
