/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/files/register', [
    'io.ox/core/extensions',
    'io.ox/core/capabilities',
    'oxguard/oxguard_core',
    'io.ox/core/folder/api',
    'oxguard/files/util',
    'io.ox/core/dropzone',
    'oxguard/auth',
    'gettext!oxguard',
    'io.ox/core/extPatterns/links'
], function (ext, capabilities, oxguard, folderAPI, util, dropzone, auth_core, gt, links) {

    'use strict';

    // For mobile, add a decrypt button
    ext.point('io.ox/files/details').extend({
        index: 500,
        id: 'downloadGuardsmall',
        draw: function (baton) {
            if (_.device('small')) {
                if (_.device('ios')) return;   // ios doesn't allow downloading.
                if (baton.data.ogfile === true) {
                    var download = $('<a href="#" class="btn btn-default">' + gt('Download decrypted') + '</a>');
                    download.click(function (ev) {
                        ev.preventDefault();
                        auth_core.authorize(baton).then(function (data) {
                            require (['oxguard/files/downloader'], function (downloader) {
                                metrics('guard', 'decrypt & download');
                                downloader.viewFile(baton, 'download', data);
                            });
                        });
                    });
                    $(this).append(download);
                }
            }
        }
    });

    ///////////  Main toolbar in drive

    //////   Links
    // Add link for add and encrypt local file
    new links.ActionLink('io.ox/files/links/toolbar/default', {
        index: 110,
        id: 'uploadEncrypt',
        label: gt('Add and encrypt local file'),
        ref: 'io.ox/files/actions/uploadEncrypt'
    });

    ext.point('io.ox/files/classic-toolbar/links').extend(new links.Link({
        id: 'remencrypt',
        index: 550,
        label: gt('Remove encryption'),
        ref: 'oxguard/remencrypt',
        mobile: 'lo',
        section: 'guard'
    }));

    ext.point('io.ox/files/classic-toolbar/links').extend(new links.Link({
        id: 'encrypt',
        index: 550,
        label: gt('Encrypt'),
        ref: 'oxguard/encrypt',
        mobile: 'lo',
        section: 'guard'
    }));

    // Mobile links
    ext.point('io.ox/files/links/inline').extend(new links.Link({
        id: 'encrypt',
        index: 1000,
        prio: 'hi',
        mobile: 'lo',
        label: gt('Encrypt'),
        ref: 'oxguard/encrypt'
    }));

    ext.point('io.ox/files/links/inline').extend(new links.Link({
        id: 'remencrypt',
        index: 1000,
        prio: 'hi',
        mobile: 'lo',
        label: gt('Remove encryption'),
        ref: 'oxguard/remencrypt'
    }));

    // Context menu

    ext.point('io.ox/core/file/contextmenu/default/items').extend(
        {
            id: 'encrypt',
            index: 1450,
            ref: 'oxguard/encrypt',
            section: '25',
            label: gt('Encrypt')
        }
    );

    ext.point('io.ox/core/file/contextmenu/default/items').extend(

        {
            id: 'decrypt',
            index: 1450,
            ref: 'oxguard/remencrypt',
            section: '25',
            label: gt('Remove encryption')
        }
    );

    // Change send copy to handle Guard emails
    ext.point('io.ox/files/actions/send').extend({
        id: 'og_stop_send',
        index: 1,
        requires: function (e) {
            if (e.collection.has('some', 'items') && !_.isEmpty(e.baton.data)) {
                var toCheck = e.baton.models === null ? e.baton.model : e.baton.models;
                if (util.hasEncrypted(toCheck)) {
                    e.stopPropagation();
                    return false;
                }
            }
        }
    });

    ext.point('io.ox/files/classic-toolbar/links').extend(new links.Link({
        id: 'sendcopy',
        index: 1200,
        prio: 'lo',
        mobile: 'lo',
        label: gt('Send by mail'),
        ref: 'oxguard/sendcopy',
        section: 'share'
    }));

    //////////////  Viewer
    /// Links

    // Extension to handle direct link files. Will not be used for opening list of files with selection
    ext.point('io.ox/core/viewer/main').extend({
        id: 'guardAuthCheck',
        index: 10,
        perform: function (baton) {
            var def = $.Deferred();
            if (baton.data.selection) {
                return def.resolve();  // Not handling here
            }
            var files = baton.data.files;
            if (!files || files.length === 0) {
                return def.resolve();
            }
            var file = files[0];
            if (isOGFile(file)) {
                auth_core.authorize(file, undefined, true)
                .then (function (auth) {
                    var file_options = { params: {
                        cryptoAction: 'Decrypt',
                        cryptoAuth: auth,
                        session: ox.session
                    } };
                    // fix for #58378
                    for (var i = 0; i < baton.data.fileList.length; i++) {
                        // fix for #58617
                        if (baton.data.fileList[i] instanceof Backbone.Model) {
                            baton.data.fileList[i].set('file_options', file_options);
                            baton.data.fileList[i].set('source', 'guardDrive');

                        } else {
                            baton.data.fileList[i].file_options = file_options;
                            baton.data.fileList[i].source = 'guardDrive';
                        }
                    }

                    def.resolve(baton);
                }, def.reject);
                return def;
            } else {
                return def.resolve();
            }
        }
    });

    ext.point('io.ox/core/viewer/actions/toolbar/popoutstandalone').extend({
        id: 'popoutstandalone_guardcheck',
        index: 1,
        requires: function (e) {
            if (e.collection.has('one')) {
                if (e.baton.model.isEncrypted()) {
                    e.stopPropagation();
                    return false;
                }
            }
        }
    });

    ext.point('io.ox/core/viewer/toolbar/links/drive').extend(new links.Link({
        id: 'GuardViewerEncrypt',
        index: 200,
        label: gt('Encrypt'),
        ref: 'oxguard/encrypt',
        mobile: 'lo'
    }));

    ext.point('io.ox/core/viewer/toolbar/links/guardDrive').extend(new links.Link({
        id: 'GuardViewerRemEncrypt',
        index: 200,
        label: gt('Remove encryption'),
        ref: 'oxguard/remencrypt',
        mobile: 'lo'
    }));

    ext.point('io.ox/core/viewer/toolbar/links/guardDrive').extend(new links.Link({
        id: 'GuardViewerDownload',
        index: 201,
        label: gt('Download decrypted'),
        ref: 'oxguard/download',
        section: 'export',
        mobile: 'lo'
    }));

    ext.point('io.ox/core/viewer/toolbar/links/guardDrive').extend(new links.Link({
        id: 'GuardViewerDownloadEncr',
        index: 202,
        label: gt('Download encrypted'),
        ref: 'oxguard/downloadEncrypted',
        section: 'export',
        mobile: 'lo'
    }));

    ext.point('io.ox/core/viewer/toolbar/links/guardDrive').extend(new links.Link({
        prio: 'hi',
        mobile: 'lo',
        label: gt('Pop out'),
        icon: 'fa  fa-external-link-square',
        ref: 'oxguard/popoutstandalone',
        customize: function () {
            this.addClass('viewer-toolbar-popoutstandalone')
                .attr({
                    tabindex: '1',
                    title: gt('Pop out standalone viewer'),
                    'aria-label': gt('Pop out standalone viewer')
                });
        }
    }));

    // Versions
    // View is removed from versions currently
    /*ext.point('io.ox/files/versions/links/inline').extend(new links.Link({
        id: 'GuardView',
        index: 90,
        label: gt('View'),
        ref: 'oxguard/versionView'
    }));
    */

    ext.point('io.ox/files/versions/links/inline').extend(new links.Link({
        id: 'GuardOpen',
        index: 100,
        label: gt('Decrypt and open'),
        ref: 'oxguard/open'
    }));

    ext.point('io.ox/files/versions/links/inline').extend(new links.Link({
        id: 'GuardDownload',
        index: 200,
        label: gt('Download decrypted'),
        ref: 'oxguard/download'
    }));

    // Remove the loc portion of the name in version list
    ext.point('io.ox/files/details/version').extend({
        index: 1,
        id: 'GuardVersion',
        draw: function () {}
    });

    //// Actions

    // Upload new encrypted file
    new links.Action('io.ox/files/actions/uploadEncrypt', {
        capabilities: 'guard-drive',
        requires: function (e) {
            return e.baton.app.folder.getData().then(function (data) {
                //hide for virtual folders (other files root, public files root)
                var virtual = _.contains(['14', '15'], data.id);
                //no new files in trash folders
                return folderAPI.can('create', data) && !virtual && !folderAPI.is('trash', data) && !folderAPI.isExternalFileStorage(data);
            });
        },
        action: function (baton) {
            ensureSetup().then(function () {
                baton.file_options = { 'params': { 'cryptoAction': 'Encrypt' }};
                ext.point('io.ox/files/actions/upload').invoke('action', this, baton);
            });
        }
    });

    new links.Action('oxguard/downloadEncrypted', {
        requires: function (e) {
            if (e.baton.data) {
                if (isOGFile(e.baton.data)) {
                    return (true);
                }
            }
            return false;
        },
        multiple: function (list) {
            list.map(function (file) {
                // For encrypted, wipe the cryptoAuth and cryptoAction
                if (file && file.params) {
                    file.params = {};
                }
            });
            ext.point('io.ox/files/actions/download').invoke('multiple', this, list);

        }
    });

    ext.point('io.ox/files/actions/open').replace({
        id: 'default',
        requires: function (e) {
            if (e.collection.has('multiple')) return false;
            if (e.baton.data) {
                if (isOGFile(e.baton.data)) {
                    return (false);
                }
            }
            // 'description only' items
            return ((e.baton.data.group !== 'localFile') && (!_.isEmpty(e.baton.data.filename) || e.baton.data.file_size > 0));
        }
    });

    // Disable publication of Guard files
    ext.point('io.ox/files/actions/getalink').extend({
        index: 10,
        id: 'guardCheck',
        requires: function (e) {
            if (isOGFile(e.baton.data)) {
                e.stopPropagation();
                return false;
            }
        }
    });

    ext.point('io.ox/files/actions/invite').extend({
        capabilities: 'guard-drive',
        index: 10,
        id: 'guardCheck',
        requires: function () {
            return false;
        },
        action: function (baton) {
            // If OG file, needs authentication to change invite permissions
            if (isOGFile(baton.data)) {
                if (baton.data && baton.data.validated) {
                    baton.data.validated = false;
                    return;  // already authorized
                }
                baton.stopPropagation();  // Terminate
                var loc = this;
                require(['oxguard/auth', 'io.ox/core/extPatterns/actions'], function (auth_core, actions) {
                    auth_core.authorize(baton, gt('Please re-enter your %s password.', window.oxguarddata.productName), true)
                    .done(function () {
                        if (baton.data) baton.data.validated = true;
                        baton.resumePropagation();
                        actions.invoke('io.ox/files/actions/invite', loc, baton);  // OK, authorized, call the extension again
                    })
                    .fail(function () {
                        baton.stopPropagation();
                        window.setTimeout(function () {
                            baton.resumePropagation();
                        }, 500);
                    });
                });
            }
        }
    });

    // Replacement popout for Guard
    new links.Action('oxguard/popoutstandalone', {
        requires: function (e) {
            if (e.collection.has('one')) {
                var currentApp = ox.ui.App.getCurrentApp().getName();
                // detail is the target of popoutstandalone, no support for mail attachments
                return (currentApp !== 'io.ox/files/detail' && e.baton.model.isEncrypted());
            }
            return false;
        },
        action: function (e) {
            ox.launch('io.ox/files/detail/main').done(function () {
                var node = this.getWindowNode();
                this.setTitle(e.data.filename);
                var app = this;
                ox.load(['io.ox/mail/actions/viewer']).done(function (action) {
                    action({ files: [e.data], app: app, container: node, standalone: true });
                });
            });
        }
    });

    // View link within versions
    new links.Action('oxguard/versionView', {
        requires: function (e) {
            return (isOGFile(e.baton.data));
        },
        multiple: function (list) {
            viewFile (list);
        }
    });

    new links.Action('oxguard/sendcopy', {
        capabilities: 'guard-drive',
        requires: function (e) {
            //all files must be encrypted
            var toCheck = e.baton.models === null ? e.baton.model : e.baton.models;
            return e.collection.has('some') && util.hasEncrypted(toCheck) && util.isNotAttachment(toCheck) && capabilities.has('webmail');
        },
        multiple: function (list) {
            list.map(function (file) {
                if (file.source === 'guardDrive') window.oxguarddata.action = 'fileAttach';
            });
            ext.point('io.ox/files/actions/send').invoke('multiple', this, list);
        }
    });

    function okToDecrypt (e, data) {
        var virtual = _.contains(['14', '15'], data.id);
        return e.collection.has('some', 'modify', 'items') && util.encryptedOnly(e.baton.models) &&
            folderAPI.can('create', data) && !virtual && !folderAPI.is('trash', data) &&
            !folderAPI.isExternalFileStorage(data);
    }

    new links.Action('oxguard/remencrypt', {
        capabilities: 'guard-drive',
        requires: function (e) {
            //all files must be encrypted
            if (e.collection.has('folders')) return false;
            if (e.baton.app) {
                return e.baton.app.folder.getData().then(function (data) {
                    return okToDecrypt (e, data);
                });
            }
            var folderId = _.isArray(e.baton.data) ? e.baton.data[0].folder_id : e.baton.data.folder_id;
            if (folderId) {
                return folderAPI.get(folderId).then(function (data) {
                    return okToDecrypt (e, data);
                });
            }
            return false;
        },
        multiple: function (list, baton) {
            ensureSetup().then(function () {
                auth_core.authorize(baton).then(function (auth) {
                    var warnShares = false;
                    for (var file in list) {
                        if (list[file].object_permissions && list[file].object_permissions.length > 0) {
                            warnShares = true;
                        }
                        list[file].file_options = {
                            params: {
                                cryptoAction: 'Decrypt',
                                cryptoAuth: auth
                            }
                        };
                    }
                    if (warnShares) {
                        require(['io.ox/core/tk/dialogs'], function (dialogs) {
                            var dialog = new dialogs.ModalDialog();
                            dialog.text(gt('Shares found'))
                                .append($('<p>').text(gt('You are removing encryption from file(s) that have shares. These shares will no longer work once the file is decrypted.  You can add the shares back once the file is decrypted.')))
                                .append($('<p>').text(gt('Do you wish to proceed?')))
                                .addPrimaryButton('decrypt', gt('Remove encryption'), 'decrypt')
                                .addButton('cancel', gt('Cancel'), 'cancel')
                                .on('decrypt', function () {
                                    doRemEncryptFile(list);
                                })
                                .show();
                        });
                    } else {
                        doRemEncryptFile(list);
                    }
                    baton.openedBy = 'io.ox/files';
                });
            });
        }
    });

    function doRemEncryptFile(list) {
        require(['io.ox/files/api'], function (fileApi) {
            fileApi.copy(list, list[0].folder_id, false)
            .then(function (e) {
                if (e.length == list.length) {
                    for (var i in list) {
                        if (e[i].data) {
                            fileApi.remove([list[i]], true);
                        }
                    }
                } else {
                    if (_.isString(e)) {
                        require (['io.ox/core/notifications'], function (notify) {
                            notify.yell('error', e);
                        });
                    }
                }
            });
        });
    }

    new links.Action('oxguard/open', {
        requires: function (e) {
            if (e.baton.data) {
                if (isOGFile(e.baton.data)) {
                    return (true);
                }
            }
            return (false);
        },
        multiple: function (list) {
            auth_core.authorize(list).then(function (auth) {
                var params = {
                    'cryptoAction': 'Decrypt',
                    'cryptoAuth': auth,
                    'session': ox.session
                };
                require (['io.ox/files/api'], function (fileApi) {
                    _(list).each(function (file) {
                        window.open(fileApi.getUrl(file, 'open', { params: params }));
                    });
                });
                metrics('guard', 'decrypt & open');
            });
        }
    });

    new links.Action('oxguard/download', {
        requires: function (e) {
            if (_.device('ios')) return false;
            var toCheck = e.baton.models === null ? e.baton.model : e.baton.models;
            if (toCheck === undefined) return false;
            return e.collection.has('some') && util.encryptedOnly(toCheck) && util.isNotAttachment(toCheck);
        },
        multiple: function (list) {
            auth_core.authorize(list).then(function (auth) {
                for (var file in list) {
                    list[file].filename = (list[file].filename ? list[file].filename.replace('.pgp', '') : '');
                    list[file].params = {
                        'cryptoAction': 'Decrypt',
                        'cryptoAuth': auth,
                        'session': ox.session
                    };
                    list[file].cache = false;
                }
                ext.point('io.ox/files/actions/download').invoke('multiple', this, list);
                metrics('guard', 'decrypt & download');
            });
        }
    });

    ////// Main listview handling

    // Call for viewer.  Check if Guard file
    ext.point('io.ox/files/actions/viewer').extend({
        id: 'guardview',
        index: 1,
        requires: function (e) {
            var toCheck = e.baton.models === null ? e.baton.model : e.baton.models;
            if (toCheck === undefined) return false;
            return e.collection.has('some') && util.encryptedOnly(toCheck);
        },
        action: function (baton) {
            if (isOGFile(baton.data)) {
                viewFile(baton);
                baton.preventDefault();
            }
        }
    });

    // Remove file decode parameters from file models once viewer closes
    ext.point('io.ox/core/viewer/toolbar/links/guardDrive').extend({
        id: 'guardCleanup',
        index: 1,
        draw: function (baton) {
            baton.context.viewerEvents.listenTo(baton.context.viewerEvents, 'viewer:beforeclose', function () {
                _.map(baton.context.collection.models, function (file) {
                    file.set('file_options', {});
                });
            });
        }
    });

    // Handle viewing Guard file
    function viewFile (baton) {
        if (isOGFile (baton.data)) {
            auth_core.authorize(baton).then(function (auth) {
                require (['oxguard/files/downloader'], function (downloader) {
                    downloader.viewFile(baton, 'view', auth);
                });
            });
        }
    }

    ext.point('io.ox/files/dropzone').extend({
        id: 'dragDrop',
        getDropZones: function (baton) {
            if (!capabilities.has('guard-drive')) return;
            var app = baton.app,
                zone2 = new dropzone.Inplace({
                    caption: gt('Drop files here to encrypt')
                });
            zone2.on({
                'show': function () {
                    app.listView.$el.stop().hide();
                },
                'hide': function () {
                    app.listView.$el.fadeIn('fast');
                },
                'drop': function (files) {
                    ensureSetup().then(function () {
                        metrics('guard', 'dropfile upload & encrypt');
                        require(['io.ox/files/upload/main'], function (fileUpload) {
                            fileUpload.setWindowNode(app.getWindowNode());
                            fileUpload.create.offer(files, { folder: app.folder.get(), 'params': { 'cryptoAction': 'Encrypt' }});
                        });
                    });
                }
            });
            return baton.dropZones.push(zone2);
        }
    });

    ////// Functions / util

    function ensureSetup() {

        if (!util.isGuardConfigured()) {
            var def = $.Deferred();
            require (['oxguard/core/createKeys'], function (keys) {
                oxguard.metrics('drive', 'create-keys');
                keys.createKeysWizard()
                .then(def.resolve, def.reject);
            });
            return def;
        }

        return $.when();
    }

    ////////  Encrypting files

    function okToEncrypt (e, data) {
        var virtual = _.contains(['14', '15'], data.id);
        return e.collection.has('some', 'modify', 'items') && !util.encryptedOnly(e.baton.models) &&
            folderAPI.can('create', data) && !virtual && !folderAPI.is('trash', data) &&
            !folderAPI.isExternalFileStorage(data);
    }

    new links.Action('oxguard/encrypt', {
        capabilities: 'guard-drive',
        requires: function (e) {
            //all files must be unencrypted
            if (e.collection.has('folders')) return false;
            if (e.baton.app) {
                return e.baton.app.folder.getData().then(function (data) {
                    return okToEncrypt (e, data);
                });
            }
            var folderId = _.isArray(e.baton.data) ? e.baton.data[0].folder_id : e.baton.data.folder_id;
            if (folderId) {
                return folderAPI.get(folderId).then(function (data) {
                    return okToEncrypt (e, data);
                });
            }
            return false;
        },
        multiple: function (list) {
            ensureSetup().then(function () {
                var warnShares = false;
                for (var file in list) {
                    if (list[file].object_permissions && list[file].object_permissions.length > 0) {
                        warnShares = true;
                    }
                    list[file].file_options = {
                        params: {
                            cryptoAction: 'Encrypt'
                        }
                    };
                }
                if (warnShares) {
                    require(['io.ox/core/tk/dialogs'], function (dialogs) {
                        var dialog = new dialogs.ModalDialog();
                        dialog.text(gt('Shares found'))
                            .append($('<p>').text(gt('There are shares associated with the file(s) you are going to encrypt. These shares will no longer work once the file is encrypted.  You can add the shares back once the file is encrypted.')))
                            .append($('<p>').text(gt('Do you wish to proceed?')))
                            .addPrimaryButton('encrypt', gt('Encrypt'), 'encrypt')
                            .addButton('cancel', gt('Cancel'), 'cancel')
                            .on('encrypt', function () {
                                doEncryptFile(list);
                            })
                            .show();
                    });
                } else {
                    doEncryptFile(list);
                }
            });

        }
    });

    function doEncryptFile(list) {
        require(['io.ox/files/api'], function (fileApi) {
            fileApi.copy(list, list[0].folder_id, false)
            .then(function (e) {
                if (e.length == list.length) {
                    for (var i in list) {
                        if (e[i].data) {
                            fileApi.remove([list[i]], true);
                        }
                    }
                } else {
                    if (_.isString(e)) {
                        require (['io.ox/core/notifications'], function (notify) {
                            notify.yell('error', e);
                        });
                    }
                }
            });
        });
    }

    /////////////////  Util, prompts, etc

    function metrics (target, action) {
        require(['io.ox/metrics/main'], function (metrics) {
            if (!metrics.isEnabled()) return;
            metrics.trackEvent({
                app: 'drive',
                target: target,
                type: 'click',
                action: action
            });
        });
    }

    /*
    function reload () {
        require(['io.ox/files/main'], function (main) {
            main.getApp().listView.reload();
        });
    }

    */
    function isOGFile(file) {
        try {
            if (!file.filename) return (false);
            if (file.meta !== undefined) {
                if (file.meta.Encrypted) return (true);
            }
            if (/(\.pgp)$/i.test(file.filename)) return true;
        } catch (e) {
            return (false);
        }
        return (false);
    }

    return {
        viewFile: viewFile
    };
});
