/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * author Greg Hill <greg.hill@open-xchange.com>
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/auth', ['oxguard/oxguard_core', 'oxguard/util',
                        'gettext!oxguard'], function (core, util, gt) {
    'use strict';

    var auth_core = {};

    // Perform auth against Guard server
    // Return auth code if any
    auth_core.authorize = function authorize(baton, prompt, minSingleUse, forceRelogin) {
        var def = $.Deferred();
        if (isFile(baton) && core.hasLocalKeys()) {  // If file and local, we need to pull the session keys, etc
            require(['pgp_local/localHandler'], function (local) {
                local.fileFound(baton, prompt, minSingleUse)
                .done(function (auth) {
                    def.resolve(auth);
                })
                .fail(function () {
                    checkAuth(prompt, minSingleUse, forceRelogin, def);
                });
            });
        } else {
            checkAuth(prompt, minSingleUse, forceRelogin, def);
        }
        return def;
    };

    function isFile (baton) {
        var file;
        if (_.isArray(baton)) {
            file = {
                data: baton[0]
            };
        } else {
            file = baton;
        }
        if (file && file.data) {
            if (file.data.filename || file.data.source == 'drive' || file.data.source == 'guardDrive') return true;
        }
        return false;
    }

    function checkAuth (prompt, minSingleUse, forceRelogin, def) {
        ensureSetup().then(function () {
            if (forceRelogin) {  // If we must relogin, show password prompt
                promptPassword (prompt, minSingleUse, def);
            } else { // Else check if we have authentication already in session
                core.checkAuth()
                .done(function (data) {
                    def.resolve(data.auth);
                })
                .fail(function () {
                    promptPassword (prompt, minSingleUse, def);
                });
            }
        }, def.reject);
    }

    // Display password prompt to authorize
    function promptPassword (prompt, minSingleUse, def) {
        if (prompt === undefined) {
            prompt = gt('Enter %s security password', window.oxguarddata.productName);
        }
        core.getPassword(prompt, true)  //Get password
        .done(function (passdata) {
            if (passdata.duration > -1 || minSingleUse) {
                core.savePassword(passdata.password, passdata.duration)
                .done(function (data) {
                    def.resolve(data.auth);
                })
                .fail(function (data) {
                    handleFail(data.auth);
                    def.reject(data.auth);
                });
            } else {
                core.auth(ox.user_id, passdata.password)
                .done(function (data) {
                    if (data.auth.length > 20 && !(/Error:/i.test(data.auth))) {
                        def.resolve(data.auth);
                    } else {
                        handleFail(data.auth);
                        def.reject(data.auth);
                    }
                });
            }
        })
        .fail(function () {
            def.reject('cancel');
        });
    }

    function ensureLoaded() {
        if (window.oxguarddata.loaded) return $.when();
        var def = $.Deferred();
        var counter = 0;
        var timer = setInterval(function () {
            if (window.oxguarddata.loaded) {
                def.resolve();
                clearInterval(timer);
            }
            if (counter++ > 100) {
                def.reject();
                clearInterval(timer); // not loading
                console.error('Response timeout from Guard server');
            }
        }, 100);
        return def;
    }

    function ensureSetup() {

        var def = $.Deferred();
        ensureLoaded().then(function () {
            if (!util.isGuardConfigured()) {
                require (['oxguard/core/createKeys'], function (keys) {
                    core.metrics('drive', 'create-keys');
                    keys.createKeysWizard()
                    .then(def.resolve, def.reject);
                });
                return;
            }

            // If keys were set up by receiving email, possible user hasn't had prompt to create password.
            if (window.oxguarddata.passcode === 'Password Needed') {
                require(['oxguard/core/tempPassword'], function (tempPass) {
                    var go = function () {
                        def.resolve();
                    };
                    tempPass.createOxGuardPasswordPrompt(null, go);
                });
            } else {
                def.resolve();
            }
        }, def.reject);

        return def;
    }

    function handleFail (auth) {
        require (['io.ox/core/notifications'], function (notifications) {
            switch (auth.toLowerCase()) {
            case 'bad password':
                notifications.yell('error', gt('Bad password'));
                return;
            case 'lockout':
                notifications.yell('error', gt('Account Locked out'));
                return;
            default:
                notifications.yell('error', gt('Unspecified error') + '\n' + auth);
                return;
            }
        });
    }

    return auth_core;
});
