/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.storage.s3.internal;

import java.net.URI;
import java.net.URISyntaxException;
import org.apache.http.client.utils.URIBuilder;
import com.openexchange.exception.OXException;
import com.openexchange.filestore.FileStorageService;
import com.openexchange.guard.storage.AbstractCoreStorage;

/**
 * Implements S3 storage for Guard.  Takes Guard formatted file locations then utilizes the middleware S3 implementation to handle
 * file actions
 * {@link S3StorageImpl}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v7.10.1
 */
public class S3StorageImpl extends AbstractCoreStorage {

    private final String bucket;
    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(S3StorageImpl.class);

    public S3StorageImpl (FileStorageService fileStorageService, String bucket) {
        this.fileStorageService = fileStorageService;
        this.bucket = bucket;
    }

    /**
     * Pull an S3 URI from the file location
     * @param location
     * @return
     */
    @Override
    protected URI uriFromLocation (String location) {
        String path = (location.indexOf("/") < 0) ? "" : location.substring(0, location.lastIndexOf("/"));
        try {
            return new URIBuilder().setScheme("s3").setHost(bucket).setPath(path).build();
        } catch (URISyntaxException e) {
            LOG.error("Problem parsing S3 URI from file location", e);
        }
        return null;
    }

    /**
     * Pull an S3 URI from the directory
     * @param directory
     * @return
     */
    @Override
    protected URI uriFromDirectory (String directory) {
        try {
            return new URIBuilder().setScheme("s3").setHost(bucket).setPath(directory).build();
        } catch (URISyntaxException e) {
            LOG.error("Problem parsing S3 URI from file directory", e);
            return null;
        }
    }


    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.Storage#objectExists(java.lang.String)
     */
    @Override
    public boolean objectExists(String objectId) throws OXException {
        // No speed improvement here for S3.  Just try file action
        return true;
    }



}
