/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp.signing.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.SignatureException;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.mail.Address;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimeUtility;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.pgp.signing.impl.SignatureResults;
import com.openexchange.guard.mime.services.MimeParser;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.PGPResult;
import com.openexchange.guard.pgp.PGPUtils;
import com.openexchange.guard.pgp.exceptions.GuardPGPExceptionCodes;
import com.openexchange.guard.pgp.signing.EmailSignatureVerificationResult;
import com.openexchange.guard.pgp.signing.GuardEmailSigningService;


/**
 * Default implementation of {@link GuardEmailSignignService}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.4.0
 */
public class GuardEmailSigningServiceImpl implements GuardEmailSigningService {

    private static Logger LOG = LoggerFactory.getLogger(GuardEmailSigningServiceImpl.class);

    private final String BeginSigned = "-----BEGIN PGP SIGNED MESSAGE-----";

    /**
     * Retrieves the mime email from the OX backend
     *
     * @param cookie The user's cookie
     * @param session The session
     * @param userAgent The client's user agent
     * @param folder The folder of the email
     * @param emailid The ID of th email
     * @return The mime mail retrieved from the backend
     * @throws MessagingException
     * @throws IOException
     */
    private MimeMessage getMimeEMail(OxCookie cookie, String session, String userAgent, String folder, String emailid) throws MessagingException, IOException {
        String mime = new Api(cookie, session, userAgent).getMime(emailid, folder);
        Session s = Session.getDefaultInstance(new Properties());
        try (InputStream in = new ByteArrayInputStream(mime.getBytes("UTF-8"))) {
            MimeMessage message = new MimeMessage(s, in);
            message.saveChanges();
            return message;
        }
    }

    /**
     * Extracts the PGP inline signed data from a mime message
     *
     * @param message The message containing the PGP inline signature
     * @param attachmentIndex The ID of the attachment representing the PGP inline signed content
     * @return The whole PGP signed message Inline block
     * @throws IOException
     * @throws MessagingException
     */
    private String extractInlineContentFromMime(MimeMessage message, int attachmentIndex) throws IOException, MessagingException {
        Object body = message.getContent();
        if (body instanceof Multipart || body instanceof MimeMultipart) {
            Multipart mp = (Multipart) body;
            BodyPart bp = mp.getBodyPart(attachmentIndex);
            String[] encoding = bp.getHeader("Content-Transfer-Encoding");
            if (encoding != null) {
                String encoded = encoding[0];
                InputStream encodedStream = bp.getInputStream();
                return getDecoded(encodedStream, encoded);
            } else {
                ByteArrayOutputStream writer = new ByteArrayOutputStream();
                mp.getBodyPart(attachmentIndex).writeTo(writer);
                return MimeUtility.decodeText(writer.toString("UTF-8"));
            }
        } else {
            if (body instanceof String) {
                return (String) body;
            }
        }
        return null;
    }

    /**
     * For Encoded data from Mime message, get the decoded (remove quoted-prinatble, etc)
     * @param encodedStream
     * @param encoding
     * @return
     * @throws MessagingException
     * @throws IOException
     */
    private String getDecoded(InputStream stream, String encoding) throws MessagingException, IOException {
        //InputStream decoded = MimeUtility.decode(encodedStream, encoding);    // InputStream already outputs decoded
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        IOUtils.copy(stream, out);
        stream.close();
        return new String(out.toByteArray(), "UTF-8");
    }

    /**
     * Extracts the PGP MIME signed data from a mime message
     *
     * @param message The message containing the PGP MIME signature
     * @return The extracted PGP MIME signed data
     * @throws IOException
     * @throws MessagingException
     */
    private String extractSignedDataFromMime(MimeMessage message) throws IOException, MessagingException {
        if (message.getContent() instanceof Multipart) {
            Multipart mp = (Multipart) message.getContent();
            for (int i = 0; i < mp.getCount(); i++) {
                try (ByteArrayOutputStream writer = new ByteArrayOutputStream()) {
                    mp.getBodyPart(i).writeTo(writer);
                    String partData = writer.toString("UTF-8");

                    if (!mp.getBodyPart(i).getContentType().contains("application/pgp-signature")) {
                        return normalize(MimeParser.getMimeSection(partData, MimeParser.getBoundry(message)));
                    }
                }
            }
        }
        return null;
    }

    /**
     * Remove any whitespaces and normalize with \r\n after each line
     *
     * @param content
     * @return
     */
    private String normalize(String content) {
        StringBuilder sb = new StringBuilder();
        String[] lines = content.split("\n");
        for (String line : lines) {
            if (line.startsWith("- -")) {
                line = line.substring(2);
            }
            sb.append(StringUtils.stripEnd(line, null) + "\r\n");
        }
        return (sb.toString());
    }

    /**
     * Extracts the PGP Mime signature from a Mime message
     *
     * @param message The message to extract the PGP mime signature from
     * @return The PGP Mime extracted signature
     * @throws IOException
     * @throws MessagingException
     */
    private String extractMimeSignatureFromMime(MimeMessage message) throws IOException, MessagingException {
        if (message.getContent() instanceof Multipart) {
            Multipart mp = (Multipart) message.getContent();
            for (int i = 0; i < mp.getCount(); i++) {
                ByteArrayOutputStream writer = new ByteArrayOutputStream();
                mp.getBodyPart(i).writeTo(writer);
                String partData = writer.toString("UTF-8");
                if (mp.getBodyPart(i).getContentType().contains("application/pgp-signature")) {
                    if (partData.contains("Encoding: base64")) {
                        return decodeBase64(partData);
                    } else {
                        return partData;
                    }
                }
            }
        }
        return null;
    }

    private String decodeBase64(String data) {
        if (data.contains("\r\n\r\n")) {
            data = data.substring(data.indexOf("\r\n\r\n"));
        }
        String ret = data;
        try {
            ret = new String(Base64.decode(data.trim()));
        } catch (Exception e) {
            LOG.error("Problem decoding Base64 attachment", e);
        }
        return ret;
    }

    /**
     * Extracts the PGP inline signature from an inline Message
     *
     * @param message The inline message
     * @return The extracted PGP inline signature
     */
    private String extractInlineSignatureFromInline(String message) {
        final String regex = "(-----BEGIN PGP SIGNATURE-----[\\s\\S]*?-----END PGP SIGNATURE-----)";
        Pattern pattern = Pattern.compile(regex, Pattern.DOTALL);
        Matcher matcher = pattern.matcher(message);
        boolean signatureFound = matcher.find();
        if (signatureFound) {
            return cleanupSignature(matcher.group(1));
        }
        return null;
    }

    /**
     * Remove any HTML markup from the actual signature data
     * @param sig
     * @return
     */
    private String cleanupSignature (String sig) {
        StringBuilder sb = new StringBuilder();
        String[] lines = sig.split("\r");
        for (String line: lines) {
            sb.append(line.trim().replaceAll("<br>", "").replaceAll("[^\\x20-\\x7E]", ""));
            sb.append("\r");
        }
        sb.append("\r");
        return (sb.toString());
    }

    /**
     * Extracts the signed PGP inline data from an inline Message
     *
     * @param message the inline message
     * @return The extracted PGP inline signed data
     */
    private String extractInlineMessageFromInline(String message) {
        if (message.contains("<br>")) {
            message = cleanupHTML(message);
        }
        final String regex = "-----BEGIN PGP SIGNED MESSAGE-----.*?\nHash:.*?\n.*?\n(.*).\\s-----BEGIN PGP SIGNATURE-----";
        Pattern pattern = Pattern.compile(regex, Pattern.DOTALL);
        Matcher matcher = pattern.matcher(message);
        boolean messageFound = matcher.find();
        if (messageFound) {
            return matcher.group(1);
        }
        return null;
    }

    /**
     * HTML emails have additional markup that often fail inline signatures
     * Try to remove here
     * @param data
     * @return
     */
    private String cleanupHTML (String data) {
        StringBuilder sb = new StringBuilder();
        String[] lines = data.split("\r");
        int indentation = 0;  // HTML often indented
        for (String line: lines) {
            if (line.contains("-----BEGIN")) {
                indentation = line.indexOf("-----BEGIN");
            }
            line = line.replaceAll("<br>", "").replaceAll("[\\xA0]", " ");  // A0 is space in extended ASCII
            line = line.substring(indentation > line.length() ? line.length() : indentation);
            if (line.startsWith("- -")) line = line.substring(2);
            sb.append(line);
            sb.append("\r\n");
        }
        return (sb.toString());
    }

    /**
     * Get the from address from a mimemessage
     * @param email
     * @return
     * @throws MessagingException
     */
    private String getFromAddress (MimeMessage email) throws MessagingException {
        Address[] from = email.getFrom();
        String fromEmail = null;
        if (from != null && from.length > 0) {
            InternetAddress addr = new InternetAddress(from[0].toString());
            fromEmail = addr.getAddress();
        }
        return fromEmail;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.pgp.signing.GuardEmailSigningService#verifySignature()
     */
    @Override
    public EmailSignatureVerificationResult verifyMimeEmailSignature(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, int userId, int contextId) throws OXException {
        try {
            MimeMessage email = getMimeEMail(cookie, session, userAgent, emailFolder, emailId);
            return verifyMimeEmailSignature(email, userId, contextId);
        } catch (IOException | MessagingException | PGPException e) {
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    private EmailSignatureVerificationResult verifyMimeEmailSignature(MimeMessage email, int userId, int contextId) throws IOException, MessagingException, OXException, PGPException {
        String signedData = extractSignedDataFromMime(email);
        if (signedData == null) {
            throw GuardPGPExceptionCodes.SIGNATURE_ERROR_NO_SIGNED_DATA.create();
        }
        String signature = extractMimeSignatureFromMime(email);
        if (signature == null) {
            throw GuardPGPExceptionCodes.SIGNATURE_ERROR_NO_SIGNATURE.create();
        }
        PGPResult pgpResult = new PGPUtils().verifySignature(signedData, signature, userId, contextId, getFromAddress(email));
        return new EmailSignatureVerificationResult(pgpResult);
    }

    @Override
    public EmailSignatureVerificationResult verifyInlineEmailSignature(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, int attachmentId, int userId, int contextId) throws OXException {
        try {
            MimeMessage email = getMimeEMail(cookie, session, userAgent, emailFolder, emailId);
            String inlineData = extractInlineContentFromMime(email, attachmentId);
            PGPResult pgpResult = verifyInlineEmailSignature (inlineData, userId, contextId, getFromAddress(email));
                if (pgpResult.isVerified()) {
                    pgpResult.setContent(getContent(email));
                }
                return new EmailSignatureVerificationResult(pgpResult);
        } catch (MessagingException | IOException e) {
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Do the verification of signature data
     * @param inlineData
     * @param userId
     * @param contextId
     * @param fromAddress
     * @return
     * @throws OXException
     */
    private PGPResult verifyInlineEmailSignature (String inlineData, int userId, int contextId, String fromAddress) throws OXException
    {
        try {
            if (inlineData == null) {
                throw GuardPGPExceptionCodes.SIGNATURE_ERROR_NO_SIGNED_DATA.create();
            }
            String signature = extractInlineSignatureFromInline(inlineData);
            if (signature == null) {
                throw GuardPGPExceptionCodes.SIGNATURE_ERROR_NO_SIGNATURE.create();
            }

            String inlineMessage = extractInlineMessageFromInline(inlineData);
            if (inlineMessage == null) {
                throw GuardPGPExceptionCodes.SIGNATURE_ERROR_NO_SIGNED_MESSAGE_FOUND.create();
            }
            try (ByteArrayInputStream signatureStream = new ByteArrayInputStream(signature.getBytes("UTF-8"))) {
                PGPResult pgpResult = new PGPUtils().verifyInlineSignature(signatureStream, inlineMessage, userId, contextId, fromAddress);
                return pgpResult;
            }
        } catch (PGPException | SignatureException | IOException e) {
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * If verified signature email, need to get clean content...removed from the PGP Signature components
     * Parses the mime message to retrieve Cleanup up content
     *
     * @param mime
     * @return
     * @throws OXException
     */
    private String getContent(MimeMessage mime) throws OXException {
        try {
            String newContent = null;
            Object content = mime.getContent();
            if (content instanceof String) {
                return (getContentInlineBody((String) content, true));
            }
            if (content instanceof Multipart || content instanceof MimeMultipart) {
                MimeMultipart mp = (MimeMultipart) content;
                int count = mp.getCount();
                for (int i = 0; i < count; i++) {
                    BodyPart bp = mp.getBodyPart(i);
                    String disp = bp.getDisposition();
                    if (disp == null || !disp.equals("attachment")) {
                        content = bp.getContent();
                        if (content instanceof String) {
                            newContent = getContentInlineBody((String) content, true);
                        }
                    }
                }
            }
            return (newContent);
        } catch (IOException | MessagingException ex) {
            LOG.error("Problem parsing inline signature for clean content ", ex);
        }
        return (null);

    }

    /**
     * Parse bodypart for content
     *
     * @param data
     * @return
     * @throws OXException
     */
    public String getContentInlineBody(String data, boolean convert) throws OXException {
        int i = data.indexOf("----BEGIN PGP SIGNATURE");
        if (i > -1) {
            data = data.substring(0, i - 1);
            int j = data.indexOf("Hash:");
            if (j > -1) {
                j = data.indexOf("\n", j) + 3;
                data = data.substring(j);
            }
            if (convert) data = data.replace("\n", "<br>");
        }
        return (data);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.pgp.signing.GuardEmailSigningService#verifyEmailSignature(javax.mail.internet.MimeMessage)
     */
    @Override
    public MimeMessage verifyEmailSignature(MimeMessage msg, int userId, int cid) throws OXException {

        try {
            if (msg.getContentType().toLowerCase().contains("pgp-signature")) {
                EmailSignatureVerificationResult result = verifyMimeEmailSignature(msg, userId, cid);
                msg.addHeader("X-Guard-Signature-Result", SignatureResults.toHeader(result));
            } else {
                verifyInlineMessage (msg, userId, cid);
            }
        } catch (IOException | MessagingException | PGPException e) {
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
        return msg;
    }

    /**
     * Verify inline signed text message
     * Remove signature from final email
     * If HTML fails signature, then remove from message
     * @param msg
     * @param userId
     * @param cid
     * @return
     * @throws OXException
     */
    private MimeMessage verifyInlineMessage (MimeMessage msg, int userId, int cid) throws OXException {
        try {
            Object content = msg.getContent();
            if (content instanceof String) {  // Simple text only email
                String data = getDecoded(msg.getInputStream(), msg.getEncoding());
                PGPResult result = verifyInlineEmailSignature(normalize(data), userId, cid, getFromAddress(msg));
                if (result.isVerified()) {  // If verified, clean up content
                    msg.setContent(getContentInlineBody((String) content, false), msg.getContentType());
                }
                EmailSignatureVerificationResult res = new EmailSignatureVerificationResult(result);
                msg.addHeader("X-Guard-Signature-Result", SignatureResults.toHeader(res));
            } else {  // Multipart.  Check for any messages for display.  All must have signatures
                if (content instanceof MimeMultipart) {
                    MimeMultipart mp = (MimeMultipart) content;
                    // Create a new messsage to return with the signature markup removed
                    MimeMultipart newpart = new MimeMultipart(mp.getContentType().contains("alternative") ? "alternative" : "mixed");
                    for (int i = 0; i < mp.getCount(); i++) {
                        BodyPart bp = mp.getBodyPart(i);
                        if (bp.getContent() instanceof String && (bp.getDisposition() == null || bp.getDisposition().contains("inline"))) {
                            String[] encoding = bp.getHeader("Content-Transfer-Encoding");
                            String data = "";
                            if (encoding != null) {
                                data = normalize(getDecoded(bp.getInputStream(), encoding[0]));
                            } else {
                                data = (String) bp.getContent();
                            }
                            try {
                                PGPResult result = verifyInlineEmailSignature(data, userId, cid, getFromAddress(msg));
                                EmailSignatureVerificationResult res = new EmailSignatureVerificationResult(result);
                                msg.addHeader("X-Guard-Signature-Result", SignatureResults.toHeader(res));
                                if (result.isVerified()) {  // If verified, clean up content
                                    bp.setContent(getContentInlineBody((String) bp.getContent(), false), bp.getContentType());
                                    newpart.addBodyPart(bp);
                                } else {
                                    if (bp.getContentType().contains("text/plain")) {  // If the plaintext fails, we'll just fail the entire thing and return original message
                                        msg.saveChanges();
                                        return msg;
                                    }
                                }
                            } catch (OXException ex) {
                                LOG.error("Problem checking inline signature in email", ex);
                                return msg;
                            }
                        } else {
                            // If not a text part or attachment, just add it back to the multipart
                            newpart.addBodyPart(bp);
                        }
                    }
                    msg.setContent(newpart);
                }
            }
            msg.saveChanges();
            return msg;
        } catch (MessagingException | IOException e) {
            return msg;
        }
    }
}
