/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.auth;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.user.GuardCapabilities;

/**
 * {@link AuthUtils}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class AuthUtils {

    private static Logger LOG = LoggerFactory.getLogger(AuthUtils.class);

    /**
     * Get the {@link GuardProperty.secureReply} 'com.openexchange.guard.SecureReply' property for the specified user in the specified context
     *
     * @param userId The user identifier
     * @param contextId The context identifier
     * @return The value of the {@link GuardProperty.secureReply} 'com.openexchange.guard.SecureReply' property
     */
    private static boolean secureReply(int userId, int contextId) {
        try {
            GuardConfigurationService guardConfigurationService = Services.getService(GuardConfigurationService.class);
            return guardConfigurationService.getBooleanProperty(GuardProperty.secureReply, userId, contextId);
        } catch (OXException e) {
            LOG.error("{}", e);
        }
        return GuardProperty.secureReply.getDefaultValue(Boolean.class);
    }


    /**
     * Create JSON that gets encrypted and returned as the auth code
     *
     * @param json
     * @param userid
     * @param cid
     * @param email
     * @return
     */
    public static JsonObject createAuthJson (int userid, int cid, String email, String authenticationFieldName, String authenticationFieldValue, String language) {
        JsonObject auth = new JsonObject();
        auth.addProperty(authenticationFieldName, authenticationFieldValue);
        auth.addProperty("email", email);
        auth.addProperty("user_id", userid);
        auth.addProperty("cid", cid);
        auth.addProperty("language", language);
        return (auth);
    }

    /**
     * Create a json of settings, include the individuals settings, plus global
     *
     * @param settings
     * @return
     * @throws OXException
     */
    public static JsonObject getSettings(String settings, GuardCapabilities setting, int id, int cid) throws OXException {
        StringBuilder allSettings = new StringBuilder();
        if (settings == null) {
            settings = "";
        }

        // If settings string loaded with key, convert to JSON
        if (!settings.startsWith("{")) {
            allSettings.append("{");
        }
        if (settings != null) {
            allSettings.append(settings);
        }
        if (!settings.endsWith("}")) {
            allSettings.append("}");
        }

        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);

        // Convert to Json, then check if global defaults present and add. Individual settings override
        Gson gson = new GsonBuilder().create();
        JsonObject json = gson.fromJson(allSettings.toString(), JsonObject.class);
        if (!json.has("Private")) {// If not defined at user level database, use global settings
            if (setting.isPrivatepass() == false) {// If config cascade value set as false, then use
                json.add("ChangePrivate", new JsonPrimitive("false"));
            } else {
                json.add("ChangePrivate", new JsonPrimitive(configService.getBooleanProperty(GuardProperty.okPrivate) ? "true" : "false"));
            }
        }

        if (!json.has("oxguard")) {
            json.add("oxguard", new JsonPrimitive(setting.hasPermission(GuardCapabilities.Permissions.MAIL)));
        }

        // If either the global setting is set for no delete keys, or the config cascade, then use
        json.add("noDeletePrivate", new JsonPrimitive(setting.isNoDeletePrivate()));
        json.add("noDeleteRecovery", new JsonPrimitive(setting.isNoDeleteRecovery()));

        json.add("noRecovery", new JsonPrimitive(setting.isNorecovery()));

        if (!configService.getBooleanProperty(GuardProperty.showStatus)) {
            json.addProperty("status", false);
        }
        json.addProperty("min_password_length", configService.getIntProperty(GuardProperty.minPasswordLength));
        json.addProperty("password_length", configService.getIntProperty(GuardProperty.newPassLength));

        json.addProperty("secureReply", secureReply(id, cid));

        return (json);
    }

}
