/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgpcore.services.impl;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Scanner;
import java.util.regex.Pattern;
import com.openexchange.exception.OXException;
import com.openexchange.guard.pgpcore.services.PGPPacketService;
import com.openexchange.guard.pgpcore.services.exceptions.PGPCoreServicesExceptionCodes;
import com.openexchange.pgp.core.exceptions.PGPCoreExceptionCodes;
import com.openexchange.pgp.core.packethandling.ExtractSessionProcessorHandler;
import com.openexchange.pgp.core.packethandling.ExtractSessionProcessorHandler.EncryptedSession;
import com.openexchange.pgp.core.packethandling.PacketProcessor;

/**
 * {@link PGPPacketServiceImpl}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.10.0
 */
public class PGPPacketServiceImpl implements PGPPacketService {

    private final Pattern       pgpPattern;
    private static final int    STREAM_INSPECTION_SIZE   = 512;
    private static final String BEGIN_PGP_MESSAGE_MARKER = "-----BEGIN PGP MESSAGE-----";

    /**
     * Initializes a new {@link PGPPacketServiceImpl}.
     */
    public PGPPacketServiceImpl() {
        this.pgpPattern = Pattern.compile(
            "-----BEGIN PGP MESSAGE-----.*?-----END PGP MESSAGE-----",
            Pattern.DOTALL);
    }

    /**
     * Encodes a given collection of session packets to a PGP conform byte represention.
     *
     * @param sessionPackets The session packets to encode.
     * @return The session packets encoded in byte representation
     * @throws IOException
     */
    private Collection<byte[]> toByteCollection(Collection<EncryptedSession> sessionPackets) throws IOException {
        ArrayList<byte[]> ret = new ArrayList<byte[]>();
        for (EncryptedSession sessionPacket : sessionPackets) {
            ret.add(sessionPacket.getEncoded());
        }
        return ret;
    }

    private boolean containsASCIIArmor(InputStream inputStream) throws IOException {
        inputStream.mark(STREAM_INSPECTION_SIZE);
        byte[] peekedData = new byte[STREAM_INSPECTION_SIZE];
        int read = inputStream.read(peekedData);
        inputStream.reset();
        if (read > 0) {
            String peekedContent = new String(peekedData, StandardCharsets.UTF_8);
            return peekedContent.contains(BEGIN_PGP_MESSAGE_MARKER);
        }

        return false;
    }

    /**
     * Internal method to extract the PGP data from the given InputStream
     *
     * @param inputStream The Inputstream to extract the PGP data from
     * @return An InputStream to the pgp subset data, extracted from the given InputStream.
     * @throws IOException
     */
    private InputStream extractPGPDataFrom(InputStream inputStream) throws IOException {
        if(!inputStream.markSupported()) {
           inputStream = new BufferedInputStream(inputStream, STREAM_INSPECTION_SIZE);
        }
        if (containsASCIIArmor(inputStream)) {
            try (Scanner scanner = new Scanner(inputStream)) {
                String pgpData = scanner.findWithinHorizon(pgpPattern, 0);
                if (pgpData != null && !pgpData.isEmpty()) {
                    return new ByteArrayInputStream(pgpData.getBytes());
                }
                return null;
            }
        } else {
            //non ASCII PGP data
            return inputStream;
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.pgpcore.services.PGPPacketService#getEncrytedSessions()
     */
    @Override
    public Collection<byte[]> getEncrytedSessions(InputStream pgpStream) throws OXException {

        try {
            pgpStream = extractPGPDataFrom(pgpStream);
        } catch (IOException e1) {
            PGPCoreExceptionCodes.IO_EXCEPTION.create(e1, e1.getMessage());
        }

        if (pgpStream != null) {
            try {
                final PacketProcessor packetProcessor = new PacketProcessor();
                final ExtractSessionProcessorHandler handler = new ExtractSessionProcessorHandler();
                packetProcessor.process(pgpStream, handler);
                return toByteCollection(handler.getEncryptedSessions());
            } catch (Exception e) {
                throw PGPCoreServicesExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
            }
        } else {
            throw PGPCoreExceptionCodes.NO_PGP_DATA_FOUND.create();
        }
    }
}
