/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.helpfiles.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.internet.MimeBodyPart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.mime.helpfiles.HelpFileService;
import com.openexchange.guard.translation.GuardTranslationService;

/**
 * {@link HelpFileServiceImpl}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.8.0
 */
public class HelpFileServiceImpl implements HelpFileService {

    private static final Logger LOG = LoggerFactory.getLogger(HelpFileServiceImpl.class);
    private static final  String HELP_FILE_NAME = "help";
    private final GuardTranslationService translationService;
    private final GuardConfigurationService guardConfigurationService;

    /**
     * Initializes a new {@link HelpFileServiceImpl}.
     *
     * @param translationService The translation to use
     */
    public HelpFileServiceImpl(GuardTranslationService translationService, GuardConfigurationService guardConfigurationService) {
        this.translationService = Objects.requireNonNull(translationService, "translationService must not be null");
        this.guardConfigurationService = Objects.requireNonNull(guardConfigurationService, "guardConfigurationService must not be null");
    }

    private List<HelpAttach> getHelp(final String...languages) {
        final List<HelpAttach> helpfiles = new ArrayList<HelpAttach>();
        try {
            String externalReaderPath = guardConfigurationService.getProperty(GuardProperty.externalReaderPath);
            String oxURL = guardConfigurationService.getProperty(GuardProperty.externalEmailURL);

            for (int i = 0; i < languages.length; i++) {
                final String lang = translationService.getAvailableCode(languages[i]);
                String data = translationService.getHelpFile(lang);
                if (data != null) {
                    data = data.replace("%guest%", "https://" + externalReaderPath);
                    data = data.replace("%webmail%", "https://" + oxURL);
                    if (!contains(helpfiles, lang)) {
                        final HelpAttach att = new HelpAttach(lang, data);
                        helpfiles.add(att);
                    }
                }
            }
        } catch (final Exception e) {
            LOG.error("Problem loading help file ", e);
        }
        return (helpfiles);
    }

    private boolean contains(final List<HelpAttach> list, final String lang) {
        for (final HelpAttach att : list) {
            if (att.getLanguage().equals(lang)) {
                return (true);
            }
        }
        return (false);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.mime.helpfiles.HelpFileService#getHelpFiles(java.lang.String[])
     */
    @Override
    public List<BodyPart> getHelpFiles(String...languages) throws OXException {
        final List<BodyPart> ret = new ArrayList<BodyPart>();
        final List<HelpAttach> attachments = getHelp(languages);
        for (int i = 0; i < attachments.size(); i++) {
            final BodyPart help = new MimeBodyPart();
            try {
                help.setFileName(translationService.getTranslation(HELP_FILE_NAME, attachments.get(i).getLanguage()) + ".txt");
                help.setContent(attachments.get(i).getTranslated(), "text/plain; charset=utf-8");
                help.setDisposition("inline");
                ret.add(help);
            }
            catch(MessagingException e) {
                throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
            }
        }
        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.mime.helpfiles.HelpFileService#getHelpFiles(java.util.Locale[])
     */
    @Override
    public List<BodyPart> getHelpFiles(Locale... locales) throws OXException {
        return getHelpFiles(Arrays.stream(locales).map(locale -> locale.toString()).toArray(String[]::new));
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.mime.helpfiles.HelpFileService#getHelpFiles(java.util.ArrayList)
     */
    @Override
    public List<BodyPart> getHelpFiles(ArrayList<RecipKey> recips) throws OXException {
        final boolean attachHelpFiles = guardConfigurationService.getBooleanProperty(GuardProperty.attachHelpFile);
        final String[] recipLanguages = recips.stream().map( recip -> recip.getLang()).toArray(String[]::new);
        return attachHelpFiles ? getHelpFiles(recipLanguages) : null;
    }
}
