/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *     @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 *
 */

package com.openexchange.guard.keymanagement.services.impl;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.IDNUtil;
import com.openexchange.guard.common.util.LongUtil;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.ImportSystemPublicKey;
import com.openexchange.pgp.keys.parsing.KeyRingParserResult;
import com.openexchange.pgp.keys.parsing.PGPKeyRingParser;
import com.openexchange.guard.keymanagement.services.impl.osgi.Services;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.keymanagement.storage.PGPKeysStorage;
import com.openexchange.guard.ratifier.GuardRatifierService;

/**
 * {@link ImportSystemPublicKeyImpl}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class ImportSystemPublicKeyImpl implements ImportSystemPublicKey {

    private String LoadFile (String file) throws OXException {
        StringBuilder sb = new StringBuilder();
        Path path = Paths.get(file);
        try (BufferedReader reader = Files.newBufferedReader(path, StandardCharsets.UTF_8)) {
            String line = null;
            while ((line = reader.readLine()) != null) {
              //process each line in some way
              sb.append(line);
              sb.append("\r");
            }
        } catch (IOException e) {
            throw OXException.general("Error loading file", e);
        }
        return sb.toString();
    }

    private String parseEmailAdress(String identity) throws AddressException, UnsupportedEncodingException {
        if (identity.contains("<")) {
            String name = identity.substring(0, identity.indexOf("<"));
            if (!name.contains("\"")) {
                identity = "\"" + name.trim() + "\" " + identity.substring(identity.indexOf("<"));
            }
        }

        InternetAddress[] recips = InternetAddress.parse(IDNUtil.aceEmail(identity), false);
        if (recips.length > 0) {
            return new InternetAddress(recips[0].getAddress(), recips[0].getPersonal(), "UTF-8").getAddress();
        }
        return null;
    }

    private void importKey (PGPPublicKeyRing publicKeyRing) throws OXException {
        ArrayList<String> keyEmailIdentities = new ArrayList<String>();
        ArrayList<String> keyHexIds = new ArrayList<String>();
        for (PGPPublicKey publicKey : publicKeyRing) {
            Iterator iter = publicKey.getUserIDs();
            while (iter.hasNext()) {
                try {
                    String address = parseEmailAdress((String) iter.next());
                    if (address != null) {
                        keyEmailIdentities.add(IDNUtil.decodeEmail(address));
                    }
                } catch (AddressException | UnsupportedEncodingException e) {
                    continue;
                }
            }
            keyHexIds.add(LongUtil.longToHexStringTruncated(publicKey.getKeyID()));
        }


        GuardRatifierService guardRatifierService = Services.getService(GuardRatifierService.class);

        if (!keyHexIds.isEmpty()) {
            for (String keyEmailIdentity : keyEmailIdentities) {
                guardRatifierService.validate(keyEmailIdentity);
                GuardKeys key = new GuardKeys();
                key.setPGPPublicKeyRing(publicKeyRing);
                key.setEmail(keyEmailIdentity);
                key.setContextid(0);
                insertKey (key);
            }
        }
    }

    private void insertKey (GuardKeys key) throws OXException {
        KeyTableStorage keyStorage = Services.getService(KeyTableStorage.class);
        int next = keyStorage.getMinFromSystemTable() - 1;
        key.setUserid(next);
        keyStorage.insert(key, true);
        EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);
        ogEmailStorage.insertOrUpdate(key.getEmail(), key.getContextid(), key.getUserid(), 0);

        //Adding the new key to the public key index (lookup table)
        PGPKeysStorage keysStorage = Services.getService(PGPKeysStorage.class);
        keysStorage.addPublicKeyIndex(key.getContextid(), 0L, key.getEmail(), key.getPGPPublicKeyRing());
    }

    private void importKeyRings (String data) throws OXException {
        PGPKeyRingParser pgpKeyParser = Services.getService(PGPKeyRingParser.class);
        try {
            KeyRingParserResult parseResult = pgpKeyParser.parse(data);
            List<PGPPublicKeyRing> keyRings = parseResult.toPublicKeyRings();
            for (PGPPublicKeyRing ring : keyRings) {
                importKey (ring);
            }
        } catch (IOException ex) {
            throw OXException.general("Problem parsing keys", ex);
        }

    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.keymanagement.services.ImportSystemPublicKey#importPublic(java.lang.String)
     */
    @Override
    public void importPublic(String file) throws OXException {
        String pgpPublicKeyData = LoadFile (file);
        importKeyRings (pgpPublicKeyData);
    }


}
