/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.commons.antiabuse;

import java.util.Objects;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import com.openexchange.antiabuse.AllowParameters;
import com.openexchange.antiabuse.ReportValue;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.common.servlets.utils.AntiAbuseUtils;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;

/**
 * {@link KeyAntiAbuseWrapper} is a helper around {@link GuardAntiAbuseService} which allows easier anti abuse handling for PGP keys.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class KeyAntiAbuseWrapper implements AntiAbuseWrapper {

    private final GuardAntiAbuseService antiAbuseService;
    private final PGPSecretKeyRing secretKeyRing;
    private final String salt;
    private AllowParameters allowParams;

    /**
     * Initializes a new {@link KeyAntiAbuseWrapper}.
     *
     * @param antiAbuseService The {@link GuardAntiAbuseService} to wrap.
     * @param key The user's {@link GuardKeys} to check the password against.
     * @param contextId The constext ID of the user
     * @param userId The user ID
     * @param password The password to check
     * @param remoteIp The remote IP of the request
     */
    public KeyAntiAbuseWrapper(GuardAntiAbuseService antiAbuseService,
                            GuardKeys key,
                            AllowParameters allowParams) {
        this(antiAbuseService,
             key.getPGPSecretKeyRing(),
             key.getSalt(),
             allowParams);
    }

    /**
     * Initializes a new {@link KeyAntiAbuseWrapper}.
     *
     * @param antiAbuseService The {@link GuardAntiAbuseService} to wrap.
     * @param secretKeyRing The {@link PGPSecretKeyRing} to check the password against.
     * @param contextId The constext ID of the user
     * @param userId The user ID
     * @param password The password to check
     * @param salt The salt of the key
     * @param remoteIp The remote IP of the request
     */
    public KeyAntiAbuseWrapper(GuardAntiAbuseService antiAbuseService,
                            PGPSecretKeyRing secretKeyRing,
                            String salt,
                            AllowParameters allowParams) {
        this.antiAbuseService = Objects.requireNonNull(antiAbuseService, "antiAbuseService must not be null");
        this.secretKeyRing = Objects.requireNonNull(secretKeyRing, "secretKeyRing must not be null");
        this.salt = salt;
        this.allowParams = allowParams;
    }

    @Override
    public <T> T doAction(AntiAbuseAction<T> action) throws Exception {
        if(antiAbuseService.allowLogin(allowParams)){
            if(PGPUtil.verifyPassword(secretKeyRing, allowParams.getPassword(),salt)) {
                antiAbuseService.report(AntiAbuseUtils.getReportParameter(ReportValue.SUCCESS, allowParams));
                return action.doAction();
            }
            else {
                antiAbuseService.report(AntiAbuseUtils.getReportParameter(ReportValue.FAILURE, allowParams));
                throw GuardAuthExceptionCodes.BAD_PASSWORD.create();
            }
        }
        else {
            throw GuardAuthExceptionCodes.LOCKOUT.create();
        }
    }
}
