/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.commons;

import java.util.List;

/**
 * {@link OGPGPKey} represents an PGP key which has been uploaded by a user
 *
 * Users are able to upload public PGP keys from other external, non OX Guard users, in order to securely interact with them
 */
public class OGPGPKey {

    private int userId;
    private int contextId;
    private String email;
    private List<String> keyIds;
    private String publicPGPAscData;
    private int shareLevel;
    private boolean inline;

    /**
     * Initializes a new {@link OGPGPKey}.
     *
     * @param userId The uploader's user ID
     * @param contextId the uploader's context ID
     * @param email the email related to the key
     * @param keyIds The key IDs related to the key
     * @param publicPGPAscData the key's ASCII armored PGP data
     * @param shareLevel the level of share
     * @param inline whether OX Guard should use PGP inline while using this key, or not
     */
    public OGPGPKey(int userId, int contextId, String email, List<String> keyIds, String publicPGPAscData, int shareLevel, boolean inline) {
        this.userId = userId;
        this.contextId = contextId;
        this.email = email;
        this.keyIds = keyIds;
        this.publicPGPAscData = publicPGPAscData;
        this.shareLevel = shareLevel;
        this.inline = inline;
    }

    /**
     * @return the user's/uploader's Id
     */
    public int getUserId() {
        return userId;
    }

    /**
     * @param userId the user's/uploader's Id
     */
    public void setUserId(int userId) {
        this.userId = userId;
    }

    /**
     * @return the user's/uploader's Id
     */
    public int getContextId() {
        return contextId;
    }

    /**
     * @param contextId the user's/uploader's Id
     */
    public void setContextId(int contextId) {
        this.contextId = contextId;
    }

    /**
     * @return The email related to the key
     */
    public String getEmail() {
        return email;
    }

    /**
     *
     * @param email the email related to the key
     */
    public void setEmail(String email) {
        this.email = email;
    }

    /**
     * @return the PGP key IDs related to the public key
     */
    public List<String> getKeyIds() {
        return keyIds;
    }

    public static String keyIdsAsString(List<String> keyIds) {
        String ret = "";
        for (int i = 0; i < keyIds.size(); i++) {
            if(i > 0) {
                ret += " ";
            }
            ret += keyIds.get(i);
        }
        return ret;
    }

    public String getKeyIdsAsString() {
        return keyIdsAsString(this.keyIds);
    }

    /**
     * @param keyIds the PGP key IDs related to the public key
     */
    public void setKeyIds(List<String> keyIds) {
        this.keyIds = keyIds;
    }

    /**
     * @return the public key's ASCII armored PGP data
     */
    public String getPublicPGPAscData() {
        return publicPGPAscData;
    }

    /**
     * @param publicPGPAscData the public key's ASCII armored PGP data
     */
    public void setPublicPGPAscData(String publicPGPAscData) {
        this.publicPGPAscData = publicPGPAscData;
    }

    /**
     * @return the key's share level
     */
    public int getShareLevel() {
        return shareLevel;
    }

    /**
     * @param shareLevel the key's new share level
     */
    public void setShareLevel(int shareLevel) {
        this.shareLevel = shareLevel;
    }

    /**
     * The inline mode which should be used for this key
     *
     * @return whether PGP inline should be used while using this key or not
     */
    public boolean isInline() {
        return inline;
    }

    /**
     * The inline mode which should be used for this key
     *
     * @param inline whether PGP inline should be used while using this key or not
     */
    public void setInline(boolean inline) {
        this.inline = inline;
    }
}
