/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.callback;

import java.util.HashMap;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.servlets.AbstractGuardServlet;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.BasicAuthServletAuthenticationHandler;
import com.openexchange.guard.servlets.authentication.GuardAuthenticationHandler;

/**
 * Handles callbacks from the OX backend
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class CallbackServlet extends AbstractGuardServlet {

    private static final long serialVersionUID = 6518623347823371079L;
    private static final String PUT_USERDELETED_CALLBACK_ACTION = "user_delete";
    private static final String PUT_USER_CHANGED = "user_update";
    private static final String PUT_CONTEXTDELETED_CALLBACK_ACTION = "context_delete";
    private static final String CHECK_GUEST = "check_guest";

    private final HashMap<String, GuardServletAction> putActions;
    private final HashMap<String, GuardServletAction> getActions;
    private final HashMap<String, GuardServletAction> postActions;

    private static final Logger logger = LoggerFactory.getLogger(CallbackServlet.class);

    /**
     * Initializes a new {@link CallbackServlet}.
     * @throws OXException
     */
    public CallbackServlet() throws OXException {
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);

        //Protecting the servlet's actions with BASIC-AUTH
        String restApiUsername = configService.getProperty(GuardProperty.restApiUsername);
        String restApiPassword = configService.getProperty(GuardProperty.restApiPassword);
        GuardAuthenticationHandler basicAuthServletAuthentication = new BasicAuthServletAuthenticationHandler(restApiUsername, restApiPassword);
        putActions = new HashMap<String, GuardServletAction>();
        getActions = new HashMap<String, GuardServletAction>();
        postActions = new HashMap<String, GuardServletAction>();
        if(!Strings.isEmpty(restApiUsername) && !Strings.isEmpty(restApiPassword)){

            //PUT
            putActions.put(PUT_USERDELETED_CALLBACK_ACTION, new UserDeletedCallbackAction().setAuthenticationHandler(basicAuthServletAuthentication));
            putActions.put(PUT_CONTEXTDELETED_CALLBACK_ACTION, new ContextDeletedCallbackAction().setAuthenticationHandler(basicAuthServletAuthentication));
            putActions.put(PUT_USER_CHANGED, new UserUpdateCallbackAction().setAuthenticationHandler(basicAuthServletAuthentication));

            //GET - This actions are used to retrieval documents for the OX Documentconverter
            getActions.put(CHECK_GUEST, new CheckGuestAction().setAuthenticationHandler(basicAuthServletAuthentication));
        }
        else{
            logger.error("Denied OX Guard callback servlet initialization due to unset Basic-Auth configuration. Please set properties 'com.openexchangeguard.restApiUsername' and 'com.openexchangeguard.restApiPassword' appropriately.");
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.http.HttpServlet#doPut(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doPut(HttpServletRequest request, HttpServletResponse response) {
        doAction(request, response, putActions);
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) {
        doAction(request, response, getActions);
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response) {
        doAction(request, response, postActions);
    }
}
