/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.authentication;

import java.util.Enumeration;
import javax.servlet.http.HttpServletRequest;
import org.apache.http.HttpHeaders;

/**
 * {@link AuthorizationHeaderParser} parses a HTTP Authorization Header
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.8.3
 */
public class AuthorizationHeaderParser {

    /**
     * Gets specified string's ASCII bytes
     *
     * @param str The string
     * @return The ASCII bytes
     */
    private static byte[] toAsciiBytes(final String str) {
        if (null == str) {
            return null;
        }
        final int length = str.length();
        if (0 == length) {
            return new byte[0];
        }
        final byte[] ret = new byte[length];
        str.getBytes(0, length, ret, 0);
        return ret;
    }

    /**
     * Parses an authorization header
     *
     * @param request The request.
     * @return The authorization value of the first authorization header found.
     */
    public String parse(HttpServletRequest request) {
        return parse(null, request);
    }

    /**
     * Parses a specific Authorization mechanism (for example "Basic").
     *
     * @param authorizationMechanism The mechanism to parse
     * @param request The request
     * @return The authorization value of the authorization header with the given mechanism
     */
    public String parse(String authorizationMechanism, HttpServletRequest request) {
        Enumeration<?> headers = request.getHeaders(HttpHeaders.AUTHORIZATION);
        while (headers.hasMoreElements()) {
            String authorisationHeader = (String) headers.nextElement();
            String[] parsedHeader = null;
            if (authorisationHeader.contains(",")) {
                //We do also support a comma separated list of header-values in one header
                parsedHeader = authorisationHeader.split(",");
            } else {
                parsedHeader = new String[] { authorisationHeader };
            }
            for (String header : parsedHeader) {
                if (header != null) {
                    header = header.trim();
                    String authType = null;
                    String authValue = null;
                    if (header.contains(" ")) {
                        authType = header.substring(0, header.indexOf(" "));
                        authValue = header.substring(header.indexOf(" "));
                    }
                    if (authType != null && authValue != null) {
                        authType = authType.trim();
                        authValue = authValue.trim();
                        if (authorizationMechanism == null ||
                            authType.toLowerCase().equals(authorizationMechanism.toLowerCase())) {
                            byte[] decodedCredentials = org.apache.commons.codec.binary.Base64.decodeBase64(toAsciiBytes(authValue));
                            return new String(decodedCredentials);
                        }
                    }
                }
            }
        }
        return null;
    }
}
