/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp.decryption.impl;

import java.io.IOException;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.internet.MimeMultipart;


/**
 * {@link PGPMimeMessageExtractor}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class PGPMimeMessageExtractor {

    /**
     * Extracts the PGP encoded part from a MultipPart MIME message
     * @param multiPartMessage The message to extract the PGP part from
     * @return The PGP part of the message, or null if no PGP part was found
     * @throws MessagingException
     * @throws IOException
     */
    public Part extractPGPMessagePart(Multipart multiPartMessage) throws MessagingException, IOException {

        Part ret = null;
        boolean pgp = false;

        //Iterating over all parts of the MultiPart message in order to find the PGP encrypted data
        for (int i = 0; i < multiPartMessage.getCount(); i++) {
            BodyPart part = multiPartMessage.getBodyPart(i);

            Object content = part.getContent();
            String type = part.getContentType();

            if (type.contains("pgp-encrypted")) {
                pgp = true;
            }
            if (type.contains("application/octet-stream") && pgp) {
                return part;
            }

            if (content instanceof Multipart || content instanceof MimeMultipart) {
                ret = extractPGPMessagePart((Multipart) content);
                if (ret != null) {
                    return ret;
                }
            }
        }
        return null;
    }

    /**
     * Extracts the PGP encoded guest message from a MultiPart MIME message
     * @param multiPartMessage The message to extract the PGP guest message from
     * @return The PGP guest part of the message, or null if no PGP part was found
     * @throws MessagingException
     * @throws IOException
     */
    public Part extractGuestMessagePart(Multipart multiPartMessage) throws MessagingException, IOException {
        Part ret = null;

        //Iterator over all parts of the MultiPart message in order to find the PGP Guest data
        for (int i = 0; i < multiPartMessage.getCount(); i++) {

            Object content = multiPartMessage.getBodyPart(i).getContent();
            String type = multiPartMessage.getBodyPart(i).getContentType();

            if (type.contains("encrypted.asc")) {
                return (multiPartMessage.getBodyPart(i));
            }
            if (content instanceof Multipart || content instanceof MimeMultipart) {
                ret = extractGuestMessagePart((Multipart) content);
            }
            if (ret != null) {
                return (ret);
            }
        }
        return (null);
    }
}
