/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp.decryption.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.NoSuchProviderException;
import java.util.ArrayList;
import java.util.Properties;

import javax.mail.Address;
import javax.mail.MessagingException;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import org.apache.commons.io.IOUtils;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.ByteArrayBody;
import org.apache.http.entity.mime.content.StringBody;
import org.bouncycastle.openpgp.PGPException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.openexchange.antiabuse.Protocol;
import com.openexchange.antiabuse.ReportValue;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.common.servlets.utils.AntiAbuseUtils;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.file.FileTyper;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.mime.services.Attachment;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.ApiResponse;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.PGPResult;
import com.openexchange.guard.pgp.PGPUtils;
import com.openexchange.guard.pgp.decryption.GuardEMailDecryptionService;
import com.openexchange.guard.pgp.decryption.PGPDecryptionResult;
import com.openexchange.guard.pgp.decryption.PGPMimeAttachmentExtractor;
import com.openexchange.guard.pgp.exceptions.GuardPGPExceptionCodes;

/**
 * Default implementation of {@link GuardEMailDecryptionService}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class GuardEMailDecryptionServiceImpl implements GuardEMailDecryptionService {

    private static Logger LOG = LoggerFactory.getLogger(GuardEMailDecryptionServiceImpl.class);

    /**
     * Retrieves the mime email from the OX backend
     *
     * @param cookie The user's cookie
     * @param session The session
     * @param userAgent The client's user agent
     * @param folder The folder of the email
     * @param emailid The ID of the email
     * @return The mime mail retrieved from the backend
     * @throws MessagingException
     * @throws IOException
     */
    private MimeMessage getMimeEMail(OxCookie cookie, String session, String userAgent, String folder, String emailid) throws MessagingException, IOException {
        String mime = new Api(cookie, session, userAgent).getMime(emailid, folder);
        Session s = Session.getDefaultInstance(new Properties());
        try (InputStream in = new ByteArrayInputStream(mime.getBytes("UTF-8"))) {
            MimeMessage message = new MimeMessage(s, in);
            message.saveChanges();
            return message;
        }
    }

    /**
     * Retrieves inline Email from OX backend
     *
     * @param cookie The user's cookie
     * @param session The session
     * @param userAgent The client's user agent
     * @param emailFolder the folder of the email
     * @param emailId The ID of the email
     * @param attachmentId The ID of the attachment representing the Mail body
     * @return
     * @throws UnsupportedEncodingException
     */
    private String getInlineEMailContent(JsonObject json, int attachmentId) throws UnsupportedEncodingException {
        JsonObject data = json.get("data").getAsJsonObject();
        JsonObject att = data.get("attachments").getAsJsonArray().get(attachmentId).getAsJsonObject();
        String content = att.get("content").getAsString();
        // Remove surrounding HTML
        int i = content.indexOf("-----BEGIN PGP MESSAGE");
        if (i > 0) {
            content = content.substring(i);
        }
        int j = content.indexOf("-----END PGP MESSAGE");
        if (j > 0) {
            int k = content.indexOf("\r", j);
            if (k > 0) {
                content = content.substring(0, k);
            }
        }
        // Convert <br> to "\r", but not if \r already there
        content = content.replace("\r<br>", "\r").replace("<br>\r", "\r").replace("<br>", "\r");
        StringBuilder sb = new StringBuilder();
        char[] chars = content.toCharArray();
        // Remove non-ascii characters (sometimes appear in body due to encoding)
        for (char c : chars) {
            if (c < 0x7f) {
                sb.append(c);
            }
        }
        return sb.toString();
    }

    private String getInlineFrom (JsonObject json) {
        String address = null;
        try {
            JsonObject data = json.get("data").getAsJsonObject();
            if (data.get("from").isJsonArray()) {
                JsonArray fromArray = data.get("from").getAsJsonArray();
                address = fromArray.get(0).getAsJsonArray().get(1).getAsString().trim();
            }
            if (data.get("from").isJsonObject()) {
                address = data.get("from").getAsJsonArray().get(1).getAsString().trim();
            }
            return address;
        } catch (Exception ex) {
            LOG.error("Problem parsing inline message for FROM address ", ex);
        }
        return address;
    }

    //FIXME/TODO: this method should be placed in another class
    private JsonObject saveToDrive(InputStream inputStream, OxCookie cookie, String session, String userAgent, int folderId, String fileName, String description, int userId, int contextId, AttachmentSaveMode saveMode) throws IOException, OXException, NoSuchProviderException, PGPException {

        Api api = new Api(cookie, session, userAgent);
        //Create the item as json
        JsonObject json = new JsonObject();
        json.addProperty("folder_id", folderId);
        json.addProperty("description", description);
        JsonObject meta = new JsonObject();
        meta.addProperty("OrigMime", FileTyper.getFileType(fileName));
        meta.addProperty("Encrypted", saveMode == AttachmentSaveMode.ENCRYPTED);
        meta.addProperty("OwnerCid", contextId);
        meta.addProperty("OwnerId", userId);
        json.add("meta", meta);

        //Attachments
        ArrayList<Attachment> attachments = new ArrayList<Attachment>();
        Attachment file = new Attachment();
        file.setContent(IOUtils.toByteArray(inputStream));
        file.setType(FileTyper.getFileType(fileName));
        file.setFilename(fileName);
        attachments.add(file);

        //Encrypt the attachment before saving it in drive
        if (saveMode == AttachmentSaveMode.ENCRYPTED) {
            //Getting the user's current PGP key
            GuardKeyService keyService = Services.getService(GuardKeyService.class);
            GuardKeys key = keyService.getKeys(userId, contextId);
            if (key != null) {
                ArrayList<RecipKey> recipients = new ArrayList<RecipKey>();
                recipients.add(new RecipKey(key));

                //Encrypt the content
                try (ByteArrayOutputStream encryptedContent = new ByteArrayOutputStream()) {
                    fileName = fileName + ".pgp";
                    PGPUtils.encrypt(encryptedContent, file.getContent(), fileName, recipients, false, true);
                    file.setContent(encryptedContent.toByteArray());
                }
            } else {
                throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_IDS_ERROR.create(userId, contextId);
            }
        }

        MultipartEntity entity = new MultipartEntity(HttpMultipartMode.BROWSER_COMPATIBLE);
        entity.addPart("json", new StringBody(json.toString(), ContentType.DEFAULT_TEXT.getMimeType(), ContentType.DEFAULT_TEXT.getCharset()));
        int attachcount = 0;
        for (Attachment attach : attachments) {
            ContentType type = ContentType.create(attach.getType().replace(" ", ""));
            entity.addPart("file_" + attachcount++, new ByteArrayBody((attach.getEncrContent() == null) ? attach.getContent() : attach.getEncrContent(), type.getMimeType(), attach.getFilename()));
        }

        String params = "files?action=new&filename=" + URLEncoder.encode(fileName, "UTF-8") + "&session=" + session;
        String response = api.filePost(json.toString(), params, entity);
        return JsonUtil.extractObject(response);
    }


    /**
    * TODO Refactor and centralize BF handling
    * @param id the id to check
    * @throws OXException if a bruteforce attempt has been detected
    */
   private void checkForBruteForce(int userid, int cid, String password, String ip) throws OXException {
       //Check for BF
       GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
       if(antiabuse.blockLogin(AntiAbuseUtils.getAllowParameter(userid + "-" + cid, password, ip, null, null))) {
           throw GuardAuthExceptionCodes.LOCKOUT.create();
       }
   }

   /**
    * TODO Refactor and centralize BF handling
    * Adds a failed login attempt for a given id
    * @param id the id to add the attempt for
    * @throws OXException due an error
    */
   private void countBruteForceAttempt(int userid, int cid, String password, String ip) throws OXException {
       GuardAntiAbuseService badService = Services.getService(GuardAntiAbuseService.class);
       badService.report(AntiAbuseUtils.getReportParameter(ReportValue.FAILURE, userid + "-" + cid, password, ip, null, null));
   }

   private String getFrom (MimeMessage msg) throws MessagingException {
       Address[] fromArray = msg.getFrom();
       if (fromArray != null && fromArray.length > 0) {
           InternetAddress from = new InternetAddress(fromArray[0].toString());
           return from.getAddress();
       }
       return "";
   }

    @Override
    public PGPDecryptionResult decryptMimeEmail(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, int userId, int contextId, String password, String remoteIp) throws OXException {

        try {
            checkForBruteForce(userId, contextId, password, remoteIp);

            //Getting the encrypted mime email
            MimeMessage message = getMimeEMail(cookie, session, userAgent, emailFolder, emailId);

            //De-Crypting the mime message
            try (ByteArrayOutputStream decryptedMessage = new ByteArrayOutputStream()) {
                PGPMimeDecrypter decrypter = new PGPMimeDecrypter();
                PGPResult pgpResult = decrypter.decryptMessage(message, userId, contextId, password, decryptedMessage, getFrom(message));

                try (InputStream decryptedMessageStream = new ByteArrayInputStream(decryptedMessage.toByteArray())) {
                    return new PGPDecryptionResult(decryptedMessageStream, message.getContentType(), pgpResult.isSignature(), pgpResult.isVerified(), pgpResult.isMissingPublicKey(), pgpResult.getError());
                }
            }
        }
        catch(OXException e) {
            if(e.getExceptionCode() == GuardPGPExceptionCodes.BAD_PASSWORD_ERROR) {
                countBruteForceAttempt(userId, contextId, password, remoteIp);
            }
            throw e;
        }
        catch (IOException | MessagingException | PGPException e) {
            throw GuardPGPExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    @Override
    public PGPDecryptionResult decryptInlineEmail(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, int attachmentId, int userId, int contextId, String password, String remoteIp) throws OXException {
        try {
            checkForBruteForce(userId, contextId, password, remoteIp);

            //Getting the Inline Email
            String jsonString = new Api(cookie, session, userAgent).getMail(emailId, emailFolder);
            JsonObject json = JsonUtil.parseAsJsonObject(jsonString);

            String inlineEMailContent = getInlineEMailContent(json, attachmentId);
            String fromAddress = getInlineFrom(json);
            //Decrypt and return the result
            try (ByteArrayOutputStream decryptedMessage = new ByteArrayOutputStream()) {
                PGPResult pgpResult = new PGPInlineDecrypter().decryptMessage(inlineEMailContent, userId, contextId, password, decryptedMessage, fromAddress);
                try (InputStream decryptedMessageStream = new ByteArrayInputStream(decryptedMessage.toByteArray())) {
                    return new PGPDecryptionResult(decryptedMessageStream, null, pgpResult.isSignature(), pgpResult.isVerified(), pgpResult.isMissingPublicKey(), pgpResult.getError());
                }
            }
        }
        catch(OXException e) {
            if(e.getExceptionCode() == GuardPGPExceptionCodes.BAD_PASSWORD_ERROR) {
                countBruteForceAttempt(userId, contextId, password, remoteIp);
            }
            throw e;
        }
        catch (IOException | PGPException e) {
            throw GuardPGPExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.pgp.decryption.PGPMimeDecryptionService#DecryptMimeAttachment(java.lang.String, java.lang.String, java.lang.String, int, int, java.lang.String, java.io.OutputStream)
     */
    @Override
    public PGPDecryptionResult decryptMimeAttachmentByName(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, String attachmentName, int userId, int contextId, String password, String remoteIp) throws OXException {

        try {
            checkForBruteForce(userId, contextId, password, remoteIp);

            //Getting the encrypted mime email
            MimeMessage message = getMimeEMail(cookie, session, userAgent, emailFolder, emailId);

            //De-Crypting the mime message
            ByteArrayOutputStream decryptedMessage = new ByteArrayOutputStream();
            PGPMimeDecrypter decrypter = new PGPMimeDecrypter();
            PGPResult pgpResult = decrypter.decryptMessage(message, userId, contextId, password, decryptedMessage, getFrom(message));

            //Extracting the attachment from the decrypted mime message by it's name
            try (InputStream decryptedMessageStream = new ByteArrayInputStream(decryptedMessage.toByteArray())) {
                Part decryptedAttachment = new PGPMimeAttachmentExtractor().getAttachmentFromMessage(decryptedMessageStream, attachmentName);
                if (decryptedAttachment == null) {
                    throw GuardPGPExceptionCodes.MIME_ATTACHMENT_NOT_FOUND.create(attachmentName);
                }
                return new PGPDecryptionResult(decryptedAttachment.getInputStream(), decryptedAttachment.getContentType(), pgpResult.isSignature(), pgpResult.isVerified(), pgpResult.isMissingPublicKey(), pgpResult.getError());
            }
        }
        catch(OXException e) {
            if(e.getExceptionCode() == GuardPGPExceptionCodes.BAD_PASSWORD_ERROR) {
                countBruteForceAttempt(userId, contextId, password, remoteIp);
            }
            throw e;
        }
        catch (IOException | MessagingException | PGPException e) {
            throw GuardPGPExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    @Override
    public PGPDecryptionResult decryptMimeAttachmentById(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, String attachmentId, int userId, int contextId, String password, String remoteIp) throws OXException {
        try {
            checkForBruteForce(userId, contextId, password, remoteIp);

            //Getting the encrypted mime email
            MimeMessage message = getMimeEMail(cookie, session, userAgent, emailFolder, emailId);

            //De-Crypting the mime message
            ByteArrayOutputStream decryptedMessage = new ByteArrayOutputStream();
            PGPMimeDecrypter decrypter = new PGPMimeDecrypter();
            PGPResult pgpResult = decrypter.decryptMessage(message, userId, contextId, password, decryptedMessage, getFrom(message));

            //Extracting the attachment from the decrypted mime message by ID
            try (InputStream decryptedMessageStream = new ByteArrayInputStream(decryptedMessage.toByteArray())) {
                Part decryptedAttachment = new PGPMimeAttachmentExtractor().getAttachmentFromMessageID(decryptedMessageStream, attachmentId);
                if (decryptedAttachment == null) {
                    throw GuardPGPExceptionCodes.MIME_ATTACHMENT_ID_NOT_FOUND.create(attachmentId);
                }
                String ct = decryptedAttachment.getContentType();
                return new PGPDecryptionResult(decryptedAttachment.getInputStream(),decryptedAttachment.getContentType(), pgpResult.isSignature(), pgpResult.isVerified(), pgpResult.isMissingPublicKey(), pgpResult.getError());
            }
        }
        catch(OXException e) {
            if(e.getExceptionCode() == GuardPGPExceptionCodes.BAD_PASSWORD_ERROR) {
                countBruteForceAttempt(userId, contextId, password, remoteIp);
            }
            throw e;
        }
        catch (IOException | MessagingException | PGPException e) {
            throw GuardPGPExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.pgp.decryption.PGPDecryptionService#DecryptInlineAttachment()
     */
    @Override
    public PGPDecryptionResult decryptInlineAttachment(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, int attachmentId, int userId, int contextId, String password, String remoteIp) throws OXException {

        try {
            checkForBruteForce(userId, contextId, password, remoteIp);

            //Get the encrypted attachment
            Api api = new Api(cookie, session, userAgent);
            Attachment att = new Attachment();
            String contentType = null;
            try(ApiResponse response = api.getPlainAttachment(emailId, Integer.toString(attachmentId), emailFolder)){
                att.setContent(response.readContent());
                org.apache.http.Header contentTypeHeader = response.getContentType();
                if(contentTypeHeader != null) {
                    contentType = response.getContentType().getValue();
                }
            }

            try (InputStream attachmentStream = new ByteArrayInputStream(att.getContent()); ByteArrayOutputStream decryptedAttachment = new ByteArrayOutputStream()) {
                //Decrypt email attachment and return an InputStream to it
                PGPResult pgpResult = PGPUtils.decryptFile(attachmentStream, decryptedAttachment, userId, contextId, password, false/* do not close the stream */);

                return new PGPDecryptionResult(new ByteArrayInputStream(decryptedAttachment.toByteArray()), contentType, pgpResult.isSignature(), pgpResult.isVerified(), pgpResult.isMissingPublicKey(), pgpResult.getError());
            }
        }
        catch (OXException e) {
            if(e.getExceptionCode() == GuardPGPExceptionCodes.BAD_PASSWORD_ERROR) {
                countBruteForceAttempt(userId, contextId, password, remoteIp);
            }
            throw e;
        }
        catch (IOException | PGPException e) {
            throw GuardPGPExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }


    @Override
    public JsonObject decryptMimeAttachmentToDrive(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, String attachmentName, int folderId, String fileName, String description, int userId, int contextId, String password, AttachmentSaveMode saveMode, String remoteIp) throws OXException {
        try {
            checkForBruteForce(userId, contextId, password, remoteIp);
            //Decrypt the mime attachment
            PGPDecryptionResult decryptionResult = decryptMimeAttachmentByName(cookie, session, userAgent, emailFolder, emailId, attachmentName, userId, contextId, password, remoteIp);
            //Save it as new item into drive

            return saveToDrive(decryptionResult.getInputStream(), cookie, session, userAgent, folderId, fileName, description, userId, contextId, saveMode);
        }
        catch (OXException e) {
            if(e.getExceptionCode() == GuardPGPExceptionCodes.BAD_PASSWORD_ERROR) {
                countBruteForceAttempt(userId, contextId, password, remoteIp);
            }
            throw e;
        }
        catch (NoSuchProviderException | IOException | PGPException e) {
            throw GuardPGPExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.pgp.decryption.GuardDecryptionService#DecryptAttachmentToDrive()
     */
    @Override
    public JsonObject decryptInlineAttachmentToDrive(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, int attachmentId, int folderId, String fileName, String description, int userId, int contextId, String password, AttachmentSaveMode saveMode, String remoteIp) throws OXException {
        try {
            //Decrypt the inline attachment
            PGPDecryptionResult decryptionResult = decryptInlineAttachment(cookie, session, userAgent, emailFolder, emailId, attachmentId, userId, contextId, password, remoteIp);
            //Save it as new item into drive
            return saveToDrive(decryptionResult.getInputStream(), cookie, session, userAgent, folderId, fileName, description, userId, contextId, saveMode);
        } catch (NoSuchProviderException | IOException | PGPException e) {
            throw GuardPGPExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }
}
