/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.milter.internal;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import org.slf4j.Logger;
import com.openexchange.exception.OXException;
import com.openexchange.guard.milter.actions.encryptor.MilterEOBActionEncryptor;
import com.openexchange.guard.milter.actions.encryptor.MilterRcptActionEncryptor;
import com.openexchange.guard.milter.service.MilterHandlerService;
import com.openexchange.guard.milter.service.MilterResponse;
import com.openexchange.guard.milter.service.impl.ActionResponse;
import com.openexchange.guard.milter.service.impl.DataResponse;
import com.openexchange.guard.milter.service.impl.GuardMilterHandlerService;
import com.openexchange.guard.milter.service.impl.GuardMilterMail;
import com.openexchange.guard.milter.service.impl.NoResponse;
import com.openexchange.guard.milter.service.impl.StandardResponse;

/**
 * {@link ConnectionHandler}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class ConnectionHandler {

    static Logger Log = org.slf4j.LoggerFactory.getLogger(ConnectionHandler.class);

    MilterHandlerService milter;

    Boolean aborted;

    public ConnectionHandler () {
        milter = new GuardMilterHandlerService();
        milter.addRcptAction(new MilterRcptActionEncryptor());
        milter.addEOBAction(new MilterEOBActionEncryptor());
        aborted = false;
    }

    public MilterResponse handle (WritableByteChannel channel, MilterPacket packet) throws IOException {

        MilterResponse resp = null;
        try {
            switch (packet.getCommand()) {
                case MilterConstants.COMMAND_CONNECT:
                    handleConnect(packet);
                    resp = new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
                    break;
                case MilterConstants.COMMAND_HELO:
                    resp = new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
                    break;
                case MilterConstants.COMMAND_OPTNEG:
                    return (handleOpt(channel));
                case MilterConstants.COMMAND_RCPT:
                    resp = milter.rcptTo(packet);
                    break;
                case MilterConstants.COMMAND_HEADER:
                    resp = milter.header(packet);
                    break;
                case MilterConstants.COMMAND_MACRO:
                    resp = new NoResponse();
                    break;
                case MilterConstants.COMMAND_BODY:
                    resp = milter.body(packet);
                    break;
                case MilterConstants.COMMAND_QUIT:
                    return null;
                case MilterConstants.COMMAND_ABORT:
                    resp = new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
                    milter.reset();
                    break;
                case MilterConstants.COMMAND_BODYEOB:
                    resp = milter.handleEOB (packet, channel);
                    break;
                default:
                    resp = new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
                    break;
            }

        }  catch (OXException ex) {
            Log.error("Problem handling Milter ", ex);
            resp = new StandardResponse(MilterConstants.RESPONSE_CONTINUE);  // Fallback is to just continue
        }

        return resp;

    }

    private static void handleConnect (MilterPacket packet) {
        ByteBuffer data = packet.getData();
        String host = MilterUtils.getStringFromData(data);
        if (data.get() == MilterConstants.SMFIA_INET) {
            data.getShort();  //port
            String addr = MilterUtils.getStringFromData(data);
            Log.debug("Milter Connection from " + host + " IP: " + addr);
        }
    }

    private static MilterResponse handleOpt (WritableByteChannel channel) throws IOException {
        ByteBuffer optionData = ByteBuffer.wrap( new byte[] {
            0x00, 0x00, 0x00, 0x02, // version
            0x00, 0x00, 0x00, 0x00, // (byte) smfif, // SMFIF_
            0x00, 0x00, 0x00, 0x00, // (byte) smfip, // SMFIP_
        });

        return new DataResponse(MilterConstants.COMMAND_OPTNEG, optionData);
    }


}
