/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services;

import java.util.Locale;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keymanagement.commons.GuardKeys;

/**
 * {@link KeyCreationService} defines functionality for creating OX Guard Keys
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public interface KeyCreationService {

    /**
     * Gets the {@link KeyPairSource} which will be used for creating new keys.
     *
     * @return The current {@link KeyPairSource}
     */
    public KeyPairSource getKeyPairSource();

    /**
     * Sets the {@link KeyPairSource} which will be used for creating new keys.
     *
     * @param keyPairSource
     */
    public void setKeyPairSource(KeyPairSource keyPairSource);


    /**
     * Creates a new {@link GuardKeys} for a specific user using a default, implementation specific {@link KeyPairSource}.
     *
     * @param userid The ID of the user
     * @param contextid The ID Of the user's context
     * @param name The name of the user
     * @param email The email of the user
     * @param password The password used for the key
     * @param locale The language related to the user
     * @param markAsCurrent True, to set the "current" flag of the new key, false otherwise
     * @param createRecovery True in order to create a password recovery for the key, false otherwise
     * @return A new, not yet persisted, {@link GuardKeys} instance
     * @throws OXException
     */
    GuardKeys create(int userid, int contextid, String name, String email, String password, Locale locale, boolean markAsCurrent, boolean createRecovery) throws OXException;

    /**
     * Creates a new {@link GuardKeys} for a specific user using a defined {@link KeyPairSource}.
     *
     * @param keyPairSource The {@link KeyPairSource} to use for creating the key pairs of the new {@link GuardKeys} object.
     * @param userid The ID of the user
     * @param contextid The ID Of the user's context
     * @param name The name of the user
     * @param email The email of the user
     * @param password The password used for the key
     * @param locale The language related to the user
     * @param markAsCurrent True, to set the "current" flag of the new key, false otherwise
     * @param createRecovery True in order to create a password recovery for the key, false otherwise
     * @return A new, not yet persisted, {@link GuardKeys} instance
     * @throws OXException
     */
    GuardKeys create(KeyPairSource keyPairSource, int userid, int contextid, String name, String email, String password, Locale locale, boolean markAsCurrent, boolean createRecovery) throws OXException;
}
