/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.commons.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPUserAttributeSubpacketVector;

/**
 * {@link PGPMergeUtil} contains helper methods for merging PGP keys.
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class PGPMergeUtil {

    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(PGPMergeUtil.class);

    private static boolean hasSig(PGPSignature sig, ArrayList<PGPSignature> sigs) throws PGPException {
        for (int i = 0; i < sigs.size(); i++) {
            if (Arrays.equals(sig.getSignature(), sigs.get(i).getSignature())) {
                return (true);
            }
        }
        return (false);
    }

    /**
     * Merges two instances of {@link PGPPublicKey}. <code>newKey</code> will be merged into <code>existingKey</code>.
     *
     * @param newKey The {@link PGPPPublicKey} which will be merged into <code>existingKey</code>
     * @param existingKey The {@link PGPPublicKey} where <code>newKey</code> will get merged into.
     * @return true, if the merge was performed, false if both keys have equals key id and the merge was not performed.
     * @throws PGPException
     */
    private static PGPPublicKey mergeKeys(PGPPublicKey newKey, PGPPublicKey existingKey) throws PGPException {
        PGPPublicKey returnKey = newKey;

        if (existingKey.getKeyID() != newKey.getKeyID()) {
            return null;
        }

        ArrayList<PGPSignature> oldsigs = new ArrayList<PGPSignature>();
        Iterator oldsigI = existingKey.getSignatures();
        while (oldsigI.hasNext()) {
            oldsigs.add((PGPSignature) oldsigI.next());
        }

        Iterator<String> userids = existingKey.getUserIDs();
        // Loop through userids of the old certificate and see if there are any new signatures
        while (userids.hasNext()) {
            String uid = userids.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForID(uid);
            if (newsigs != null) {
                while (newsigs.hasNext()) {
                    PGPSignature newsig = newsigs.next();
                    if (!hasSig(newsig, oldsigs)) {
                        returnKey = PGPPublicKey.addCertification(existingKey, uid, newsig);
                    }
                }
            }

        }
        userids = newKey.getUserIDs();
        // Loop through userids of the NEW certificate and see if there are any new signatures
        while (userids.hasNext()) {
            String uid = userids.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForID(uid);
            while (newsigs.hasNext()) {
                PGPSignature newsig = newsigs.next();
                if (!hasSig(newsig, oldsigs)) {
                    returnKey = PGPPublicKey.addCertification(existingKey, uid, newsig);
                }
            }
        }
        // Loop through user attributes and see if any new signatures
        Iterator<PGPUserAttributeSubpacketVector> vectors = existingKey.getUserAttributes();
        while (vectors.hasNext()) {
            PGPUserAttributeSubpacketVector vector = vectors.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForUserAttribute(vector);
            while (newsigs.hasNext()) {
                PGPSignature newsig = newsigs.next();
                if (!hasSig(newsig, oldsigs)) {
                    returnKey = PGPPublicKey.addCertification(existingKey, vector, newsig);
                }
            }
        }
        // Loop through user attributes and see if any new signatures
        vectors = newKey.getUserAttributes();
        while (vectors.hasNext()) {
            PGPUserAttributeSubpacketVector vector = vectors.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForUserAttribute(vector);
            if (newsigs != null) {
                while (newsigs.hasNext()) {
                    PGPSignature newsig = newsigs.next();
                    if (!hasSig(newsig, oldsigs)) {
                        returnKey = PGPPublicKey.addCertification(existingKey, vector, newsig);
                    }
                }
            }
        }

        // Check for revoke signatures
        Iterator<PGPSignature> newsigs = newKey.getSignaturesOfType(PGPSignature.KEY_REVOCATION);
        while (newsigs.hasNext()) {
            PGPSignature newsig = newsigs.next();
            if (!hasSig(newsig, oldsigs)) {
                returnKey = PGPPublicKey.addCertification(existingKey, newsig);
            }
        }
        return returnKey;
    }

    public static PGPPublicKeyRing mergeKeyRings(PGPPublicKeyRing firstRing, PGPPublicKeyRing secondRing) {

        try {
            Iterator<PGPPublicKey> keys = firstRing.getPublicKeys();
            while (keys.hasNext()) {
                PGPPublicKey keyToImport = keys.next();
                long importID = keyToImport.getKeyID();
                PGPPublicKey toMerge = secondRing.getPublicKey(importID);
                if (toMerge != null) {
                    secondRing = PGPPublicKeyRing.removePublicKey(secondRing, toMerge);
                    toMerge = mergeKeys(keyToImport, toMerge);
                    if (toMerge == null) {
                        return null;
                    }
                    secondRing = PGPPublicKeyRing.insertPublicKey(secondRing, toMerge);
                } else {
                    secondRing = PGPPublicKeyRing.insertPublicKey(secondRing, keyToImport);
                }
            }
        } catch (Exception e) {
            logger.error("Problem merging duplicate public keys, ", e);
        }
        return secondRing;
    }
}
