/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.antiabuse.bad;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.osgi.Services;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.utils.DBUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;

/**
 * {@link GuardAntiAbuseServiceImpl} is a REST DB implementation for a BadStorage
 */
public class GuardBadStorageImpl implements GuardBadStorage {

    @Override
    public Bad getBad(String id) throws OXException {
        if (id == null || id.isEmpty()) {
            throw new IllegalArgumentException("parameter id is null or empty");
        }

        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getReadOnlyForGuard();

        PreparedStatement stmt = null;
        ResultSet resultSet = null;

        try {
            stmt = connection.prepareStatement(BadSql.SELECT_BY_ID_STMT);
            stmt.setString(1, id);
            resultSet = stmt.executeQuery();

            if (resultSet.next()) {
                return new Bad(id, resultSet.getTimestamp("last"), resultSet.getTimestamp("cur"), resultSet.getInt("count"));
            }
            return null;
        } catch (SQLException e) {
            throw GuardCoreExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(resultSet, stmt);
            guardDatabaseService.backReadOnlyForGuard(connection);
        }
    }

    @Override
    public void addBad(String id) throws OXException {
        if (id == null || id.isEmpty()) {
            throw new IllegalArgumentException("parameter id is null or empty");
        }

        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getWritableForGuard();

        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(BadSql.DELETE_BY_TIME_STMT);
            GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
            stmt.setInt(1, configService.getIntProperty(GuardProperty.badMinuteLock));
            stmt.executeUpdate();

            DBUtils.closeSQLStuff(stmt);

            stmt = connection.prepareStatement(BadSql.ADD_STMT);
            stmt.setString(1, id);
            stmt.executeUpdate();
        } catch (SQLException e) {
            throw GuardCoreExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt);
            guardDatabaseService.backWritableForGuard(connection);
        }
    }

    @Override
    public void removeBad(String id) throws OXException {
        if (id == null || id.isEmpty()) {
            throw new IllegalArgumentException("parameter id is null or empty");
        }

        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getWritableForGuard();

        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(BadSql.DELETE_BY_ID_STMT);
            stmt.setString(1, id);
            stmt.executeUpdate();
        } catch (SQLException e) {
            throw GuardCoreExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt);
            guardDatabaseService.backWritableForGuard(connection);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.database.storage.bad.BadStorage#isBad(java.lang.String, int)
     */
    @Override
    public boolean isBad(String id, int threshold) throws OXException {
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        Bad badEntry = getBad(id);
        if (badEntry != null) {
            long differential = configService.getIntProperty(GuardProperty.badMinuteLock) * 60 * 1000;
            if (badEntry.getLast().getTime() > (badEntry.getCurrentTime().getTime() - differential)) {
                if (badEntry.getCount() > threshold) {
                    return true;
                }
            }
        }
        return false;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.database.storage.bad.BadStorage#isBad(java.lang.String)
     */
    @Override
    public boolean isBad(String id) throws OXException {
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        return isBad(id,configService.getIntProperty(GuardProperty.badMinuteLock));
    }
}
