package com.openexchange.guard.support;

import java.net.URI;
import java.net.URISyntaxException;
import org.apache.http.client.utils.URIBuilder;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.model.DeletedKey;
import com.openexchange.guard.database.repositories.DeletedKeyRepository;


/**
 * Exposes a private key of a deleted user for downloading
 *
 * When user's get deleted we backup their keys for a certain amount of time.
 * In that time-span an administrator can enable download functionality for
 * a user to ensure the user can get his key, even if his account was already
 * deleted.
 */
public class DeletedKeyExposer {

    /**
     * Internal method to create an URI for accessing an exposed key
     * @param deletedKey The deleted key to create an URI for
     * @return An URI from which a user can download an exposed key
     * @throws URISyntaxException
     */
    private URI buildGetExposedKeyUri(DeletedKey deletedKey) throws URISyntaxException{

        //Reading the whole reader path from the configuration
        String fullReaderPath = Config.externalReaderPath;
        if(!fullReaderPath.contains("/")) {
            throw new IllegalArgumentException("externalReaderPath is not configured correctly: Missing \"/\".");
        }

        //extracting the host name
        int firstSlashIndex = fullReaderPath.indexOf("/");
        String hostName = fullReaderPath.substring(0, firstSlashIndex ) ;
        if(hostName.isEmpty()) {
            throw new IllegalArgumentException("externalReaderPath is not configured correctly: Missing hostname.");
        }

        //extracting the path
        String readerPath = fullReaderPath.substring(firstSlashIndex);
        if(hostName.isEmpty()) {
            throw new IllegalArgumentException("externalReaderPath is not configured correctly: Missing path.");
        }


        return new URIBuilder().
                   setScheme("http" + (Config.backend_ssl ? "s" : "")).
                   setHost(hostName).
                   setPath(readerPath).
                   setParameter("action","getKey").
                   setParameter("userid", String.valueOf(deletedKey.getUserId())).
                   setParameter("cid", String.valueOf(deletedKey.getCid())).
                   setParameter("token", String.valueOf(deletedKey.getSalt())).
                   build();
    }

    /**
     * Exposes all deleted keys for the given user
     * @param email the user's email
     * @param cid the context id
     * @throws Exception On error during exposing the key
     * @return an URI for accessing deleted but exposed keys for the user
     */
    public URI exposeKey(String email, int cid) throws Exception{

        DeletedKeyRepository deletedKeyRepo = new DeletedKeyRepository();

        //Checking if the user has deleted keys and get the salt of the first found key
        DeletedKey firstDeletedKey = deletedKeyRepo.getFirst(email, cid);
        if(firstDeletedKey == null){
                throw new Exception(
                    String.format("Could not find deleted key for email \"%s\" in context %d",
                                  email,
                                  cid));
        }

        //Creating URI for accessing deleted,exposed keys
        // example:
        //http://192.168.0.69:8337/appsuite/api/oxguard/reader/reader.html?action=getkey&userid=4&cid=1&token=mdl%2B1UEEycWnub%2BHb8yLxmhc3Yg%3D
        URI uri = buildGetExposedKeyUri(firstDeletedKey);

        //Exposing the keys of the given email in order to be available using the URI
        deletedKeyRepo.setExposed(email, cid);

        return uri;
    }
}
