/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.storage.file;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;


import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.CacheManager;

import com.openexchange.guard.config.Config;
import com.openexchange.guard.encr.EncryptedObj;
import com.openexchange.guard.storage.Storage;


/**
 * Class for saving encrypted object to and from file system.
 * 
 * @author benjamin.otterbach@open-xchange.com
 */

public class FileStorage {
	private static Logger logger = LoggerFactory.getLogger(File.class);

    /**
     * Write byte array to filename
     * 
     * @param filename Full filename of object
     * @param data byte array of data
     * @return
     * @throws IOException
     */
    public static Boolean writeObj(String directoryPrefix, String ObjId, byte[] data) throws IOException {
    	String directory = Config.getRootFileDirectory() + Storage.DELIMITER + directoryPrefix;
    	
        java.io.File file = new java.io.File(directory);
        if (!file.exists()) {
            file.mkdir();
        }
    	
        Boolean result = false;
        OutputStream output = null;
        try {
            output = new BufferedOutputStream(new FileOutputStream(directory + Storage.DELIMITER + ObjId));
            output.write(data);

            result = true;
        } catch (Exception e) {
            logger.error("Error writing output stream for file " + ObjId, e);
        } finally {
            output.close();
        }
        return (result);
    }

    /**
     * Get an object from the cache store.
     * 
     * @param directoryPrefix Sub-directory of the files storage location, for example: ext_1_25
     * @param ObjId ID of the file
     * @return EncryptedObj
     */
    public static EncryptedObj getEncrObj(String directoryPrefix, String ObjId) {
        try {

            InputStream in = new FileInputStream(Config.getRootFileDirectory() + Storage.DELIMITER + directoryPrefix + Storage.DELIMITER + ObjId);
            EncryptedObj obj = new EncryptedObj(in);
            in.close();
            return (obj);
        } catch (FileNotFoundException e) {
            logger.error("File not found");
            EncryptedObj obj = new EncryptedObj();
            obj.ItemID = "not found";
            return (obj);
        } catch (Exception ex) {
            logger.error("Error getting encrypted file " + ObjId, ex);
            return (null);
        }
    }
    
    public static InputStream getObjectStream(String directoryPrefix, String ObjId) {
        try {

            InputStream in = new FileInputStream(Config.getRootFileDirectory() + Storage.DELIMITER + directoryPrefix + Storage.DELIMITER + ObjId);
            return (in);
        } catch (FileNotFoundException e) {
            logger.error("File not found");
            return (null);
        } catch (Exception ex) {
            logger.error("Error getting encrypted file " + ObjId, ex);
            return (null);
        }
    }
    
    /**
     * Delete a file from the cache store. Usually used by the Cache Cleaner.
     * 
     * @param location File location including directory prefix. Example: ext_1_25/0ff183ed-097c-4a5a-b534-0198659c775a
     * @throws IOException
     */
    public static void deleteObj(String location) throws IOException {
    	String fileWithPath = Config.getRootFileDirectory() + Storage.DELIMITER + location;
    	
        File file = new File(fileWithPath);
        if (file.exists())
            file.delete();
        int i = fileWithPath.lastIndexOf(Storage.DELIMITER);
        File directory = new File(fileWithPath.substring(0, i));
        if (directory.isDirectory()) {
            if (directory.list().length == 0) {
                directory.delete();
            }
        }
    }

}
