package com.openexchange.guard.server.authentication;

import org.eclipse.jetty.security.ConstraintMapping;
import org.eclipse.jetty.security.ConstraintSecurityHandler;
import org.eclipse.jetty.security.HashLoginService;
import org.eclipse.jetty.security.authentication.BasicAuthenticator;
import org.eclipse.jetty.util.security.Constraint;
import org.eclipse.jetty.util.security.Credential;

/**
 * Simple factory creating authentication constraints for accessing guard
 */
public class BasicAuthSecurityHandlerFactory {

    private static final String realmName = "Guard";
    private final String[] availableRoleNames;

    /**
     * Constructor
     * @param availableRoles a list of known roles
     */
    public  BasicAuthSecurityHandlerFactory(GuardAuthRole[] availableRoles){

        availableRoleNames = new String[availableRoles.length];
        for(int i=0; i<availableRoles.length; i++) {
            availableRoleNames[i] = availableRoles[i].name();
        }
    }


    /**
     * Creates a security handler using BASIC AUTH
     * @param username the user's name
     * @param password the user's password
     * @param roles The roles to apply to the user
     * @return
     */
    public ConstraintSecurityHandler CreateSecurityHandler(String username,
                                                           Credential password,
                                                           GuardAuthRole[] roles) {

        //Setting up the constraint
        Constraint constraint = new Constraint(Constraint.__BASIC_AUTH,"user");
        constraint.setRoles(availableRoleNames);
        constraint.setAuthenticate(true);

        //Mapping of the constraint
        ConstraintMapping constraintMapping = new ConstraintMapping();
        constraintMapping.setConstraint(constraint);
        constraintMapping.setPathSpec("/*");

        //Getting role names form roles
        String[] roleNames = new String[roles.length];
        for(int i=0; i<roles.length; i++) {
            roleNames[i] = roles[i].name();
        }
        //Setting up an authentication handler
        HashLoginService hashLoginService = new HashLoginService();
        hashLoginService.setName(realmName);
        hashLoginService.putUser(username, password, roleNames);

        //Setting up the security handler
        ConstraintSecurityHandler securityHandler = new ConstraintSecurityHandler();
        securityHandler.setAuthenticator(new BasicAuthenticator());
        securityHandler.setRealmName(realmName);
        securityHandler.setConstraintMappings(new ConstraintMapping[] {constraintMapping});
        securityHandler.setLoginService(hashLoginService);
        return securityHandler;
    }
}
