/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.server;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URISyntaxException;
import java.net.URLEncoder;
import java.util.ArrayList;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.FileItemIterator;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.lang.time.StopWatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.drive.FileDecrypter;
import com.openexchange.guard.drive.FileEncrypter;
import com.openexchange.guard.encr.Crypto;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.EncryptedObj;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.GuardMissingParameter;
import com.openexchange.guard.file.Fileshare;
import com.openexchange.guard.logging.LogAction;
import com.openexchange.guard.mailcreator.Attachment;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.ox.streaming.FileItemInputStream;
import com.openexchange.guard.ox.streaming.GuardStreamUtils;
import com.openexchange.guard.ox.streaming.JsonStreamHandler;
import com.openexchange.guard.ox.streaming.StreamHandler;
import com.openexchange.guard.ox.streaming.StreamHandlerFactory;
import com.openexchange.guard.server.Settings.permissions;
import com.openexchange.guard.translate.GetText;
import com.openexchange.guard.util.Core;

public class FileHandler {

    private static Logger logger = LoggerFactory.getLogger(FileHandler.class);

    /**
     * Handling drive file uploads from a client
     *
     * @param request The request sent by the client
     * @param response The response
     * @param cookie the cookie
     * @param update ???
     * @throws IOException
     * @throws GuardMissingParameter
     */
    public void encryptUpload(HttpServletRequest request,
        HttpServletResponse response,
        OxCookie cookie,
        boolean update) throws Exception {

        //Checking if the user is allowed to use files
        if (!Auth.checkPermission(cookie,
            request,
            Settings.permissions.FILE)) {
            logger.error("not authorized");
            Core.sendFail(response, "not authorized");
            return;
        }

        String sessionID = Core.getStringParameter(request, "session", true);
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        Api api = new Api(cookie, sessionID, request.getHeader("User-Agent"));
        FileItemFactory fileItemFactory = new DiskFileItemFactory();
        ServletFileUpload servletFileUpload = new ServletFileUpload(fileItemFactory);
        String fileName = null;
        String contentType = null;
        FileItemIterator fileItemIterator = servletFileUpload.getItemIterator(request);

        try(FileItemInputStream uploadStream = new FileItemInputStream("file",fileItemIterator);
            FileItemInputStream jsonStream = new FileItemInputStream("json",fileItemIterator);)
        {
            //Getting the file name of the file to upload
            //NOTE: This opens the stream of the first item;
            //      This is OK since it is the first item and therefore
            //      we'll not skip something
            fileName = uploadStream.getItemName();
            contentType = uploadStream.getContentType();

            //Getting the user's key for encrypting the file
            //TODO: What happens when uploading a file to a shared folder?
            //      Encrypting the file for more recipients?
            //      We should move this code in it's own class then
            Access acc = new Access();
            GuardKeys userKeys = acc.getKeys(userid, cid);
            if(userKeys == null) {
                throw new Exception(String.format("Key not found for userid %d within context %d",
                                                  userid, cid));
            }

            //Adding the user to the list of recipients
            ArrayList<RecipKey> recipients = new ArrayList<RecipKey>();
            recipients.add(new RecipKey(userKeys));

            //Creating a Json handler which adds and additional
            //meta-data object to the Json stream during later
            //processing
            JsonObject meta = new JsonObject();
            meta.addProperty("Encrypted", true);
            meta.addProperty("OwnerCid", cid);
            meta.addProperty("OwnerId", userid);
            meta.addProperty("OrigMime", contentType);
            JsonStreamHandler jsonHandler = new JsonStreamHandler("meta",meta);

            //encrypting the input stream of the upload item
            logger.debug("Decrypting new upload ...");
            StopWatch sw = new StopWatch();
            sw.start();
            FileEncrypter encryptor = new FileEncrypter(api);
            JsonObject ret = encryptor.Encrypt(uploadStream,
                                               fileName,
                                               contentType,
                                               jsonStream,
                                               jsonHandler,
                                               recipients);
            sw.stop();
            logger.debug(String.format("Decryption done for new upload [Time: %d ms]",
                        sw.getTime()));

            //Sending response
            sendOK(response, ret.toString());
        }
    }

    private String getFileJson(String itemId, String mime, int folder, String description, int id, int cid) {
        Gson gson = new GsonBuilder().create();
        JsonObject json = new JsonObject();
        JsonObject meta = new JsonObject();
        meta.addProperty("OGID", itemId);
        meta.addProperty("Encrypted", true);
        meta.addProperty("OwnerCid", cid);
        meta.addProperty("OwnerId", id);
        meta.addProperty("OrigMime", mime);
        json.add("folder_id", new JsonPrimitive(folder));
        json.addProperty("description", description);
        json.add("meta", meta);
        String data = gson.toJson(json);
        return (data);
    }

    /**
     * Decode and send encrypted file
     *
     * @param request The client's request
     * @param response The response
     * @param cookie cookie data
     * @return
     * @throws IOException
     * @throws GuardMissingParameter
     * @throws URISyntaxException
     */
    public void getFile(HttpServletRequest request,
                        HttpServletResponse response,
                        OxCookie cookie) throws Exception {

        String fileId = Core.getStringParameter(request, "id", true);
        int folderId = Core.getIntParameter(request, "folder", true);
        int version = Core.getIntParameter(request, "version", true);
        String auth = Core.getStringParameter(request, "auth");
        String lang = Core.getStringParameter(request, "lang");
        int userId = Core.getIntParameter(request, "userid");
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userId, cid);
        String fileName = Core.getStringParameter(request, "filename", true);
        String encrExtra = Core.getStringParameter(request, "encrextra");
        boolean download = Core.getBooleanParameter(request, "download");

        //Get further meta data from the object to decrypt
        Api ap = new Api(cookie,request);
        String data = ap.getInfoItemJson(fileId, folderId, version);
        Gson gson = new Gson();
        JsonObject json = gson.fromJson(data, JsonObject.class);
        JsonObject jdata = json.get("data").getAsJsonObject();
        JsonObject meta = jdata.get("meta").getAsJsonObject();
        long fileSize = jdata.get("file_size").getAsLong();
        String fileType = jdata.get("file_mimetype").getAsString();
        if (meta.has("OrigMime")) {  // If we stored the original mime type in meta, use this
            fileType = meta.get("OrigMime").getAsString();
        }
        //Authentication
        UserData authData;
        if (auth.length() < 20) {
            String password = Core.getStringParameter(request, "password", true);
            authData = new UserData();
            authData.encr_password = password;
            authData.userid = userId;
        } else {
            authData = new UserData(auth, cookie);
        }
        if (authData .userid == -1) {
            sendFail(response, GetText.gettext("Unable to decode file, bad authorization", lang));
            return;
        }

        //Creating an appropriated stream handler for file decryption
        StreamHandler streamHandler =
            StreamHandlerFactory.CreateStreamHandler(fileName,
                                                     userId,
                                                     cid,
                                                     authData.encr_password,
                                                     encrExtra,
                                                     cookie.JSESSIONID);
        //Preparing response
        fileName = fileName.replace(".pgp", "").replace(".grd", "");
        String newContentType = streamHandler.getNewContentType(fileType, fileName);
        response.setContentType(newContentType);
        response.addHeader("Access-Control-Allow-Origin", "*");
        if (download) {
            response.setHeader("Content-Disposition", "attachment; filename=\"" + fileName + "\"");
        }

        OutputStream outputStream = response.getOutputStream();
        try(InputStream inputStream = ap.openDriveItem(fileId, fileSize,folderId, version))
        {
            //Just writing the input stream to the output stream of the response.
            //Encryption is done one the fly by the streamHandler
            GuardStreamUtils.Copy(inputStream,
                                  outputStream,
                                  streamHandler);
        }

        response.setStatus(HttpServletResponse.SC_OK);
    }

    /**
     * Send answer to response stream
     *
     * @param response
     * @param answer String to put in HTML
     */
    private void sendOK(HttpServletResponse response, String answer) {
        response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            response.getWriter().println(answer);
        } catch (IOException e) {
            logger.error("Error while writing 'OK' answer to response", e);
        }
    }

    /**
     * Send answer to response stream
     *
     * @param response
     * @param answer String to put in HTML
     */
    private void sendOK(HttpServletResponse response, String type, byte[] answer) {
        response.setContentType(type);

        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            OutputStream out = response.getOutputStream();
            out.write(answer);
            out.close();
        } catch (IOException e) {
            logger.error("Error while writing answer to output stream", e);
        }
    }

    private void sendFail(HttpServletResponse response, String answer) {
        response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_NOT_ACCEPTABLE);
        try {
            response.getWriter().println(answer);
        } catch (IOException e) {
            logger.error("Error while writing 'fail' answer to response", e);
        }

    }

    private void saveToDownload(HttpServletResponse response, String type, byte[] answer, String filename) {
        response.setContentType(type);
        response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            OutputStream out = response.getOutputStream();
            out.write(answer);
            out.close();
        } catch (IOException e) {
            logger.error("Error while writing to output stream", e);
        }
    }

    public void encryptFile(HttpServletRequest request,
        HttpServletResponse response,
        com.openexchange.guard.server.OxCookie cookie) throws Exception {

        String id = Core.getStringParameter(request, "id", true);
        int folder = Core.getIntParameter(request, "folder", true);
        int version = Core.getIntParameter(request, "version", true);
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        if (!Auth.checkPermission(cookie, request, permissions.MAIL)) {
            logger.error("not authorized");
            Core.sendFail(response, "not authorized");
            return;
        }

        //Get meta data from the file to encrypt
        Api api = new Api(cookie, request);
        String data = api.getInfoItemJson(id, folder, version);
        Gson gson = new Gson();
        JsonObject json = gson.fromJson(data, JsonObject.class);
        if(json.has("error")) {
            throw new Exception(json.get("error").getAsString());
        }
        JsonObject jdata = json.get("data").getAsJsonObject();
        long fileSize = jdata.get("file_size").getAsLong();
        String fileName = jdata.get("filename").getAsString();
        String fileType = jdata.get("file_mimetype").getAsString();

        //Getting the user's key
        Access acc = new Access();
        GuardKeys userKeys = acc.getKeys(userid, cid);
        if(userKeys == null) {
            throw new Exception(String.format("Key not found for userid %d within context %d",
                                              userid, cid));
        }

        //Adding the user to the list of recipients
        ArrayList<RecipKey> recipients = new ArrayList<RecipKey>();
        recipients.add(new RecipKey(userKeys));

        //populating new JSON meta data
        String itemID = EncrLib.getUUID();
        String description =
            (jdata.get("description") == null) ? "" :
                jdata.get("description").getAsString();
        String newjson = getFileJson(itemID, fileType, folder, description, userid, cid);
        JsonStreamHandler jsonHandler = new JsonStreamHandler();
        logger.debug(String.format("Starting encryption for file %s in folder %d ...",
                                    id,
                                    folder));
        StopWatch sw = new StopWatch();
        sw.start();

        try(InputStream jsonStream = new ByteArrayInputStream(newjson.getBytes());
            InputStream remoteInputStream = api.openDriveItem(id, fileSize, folder, version))
        {
            FileEncrypter fileEncrypter = new FileEncrypter(api);
            //Encrypting the item
            JsonObject retJson = fileEncrypter.Encrypt(remoteInputStream,
                                                       fileName,
                                                       fileType,
                                                       jsonStream,
                                                       jsonHandler,
                                                       recipients);
            sw.stop();
            long ms = sw.getTime();
            logger.debug(String.format("Encryption done for file %s in folder %d [Time: %d ms]",
                         id,
                         folder,
                         ms));

            //deleting the old item
            api.deleteItem(id, folder);

            //sending response
            sendOK(response, retJson.toString());
        }
    }

    /**
     * Removes the encryption from a drive file
     *
     * @param request The request sent by a client
     * @param response The response
     * @param cookie The cookie data
     * @throws GuardMissingParameter If the requests was not correct and complete
     */
    public void removeEncryption(HttpServletRequest request,
        HttpServletResponse response,
        OxCookie cookie) throws Exception {

        //parse parameters
        String fileId = Core.getStringParameter(request, "id", true);
        int folderId = Core.getIntParameter(request, "folder", true);
        int version = Core.getIntParameter(request, "version", true);
        int userId = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        String auth = Core.getStringParameter(request, "auth");
        String lang = Core.getStringParameter(request, "language");
        String fileName = Core.getStringParameter(request, "filename", true);
        String encrExtra = Core.getStringParameter(request, "encrextra");

        //Get further meta data from the object to decrypt
        Api ap = new Api(cookie, request);
        String data = ap.getInfoItemJson(fileId, folderId, version);
        Gson gson = new Gson();
        JsonObject json = gson.fromJson(data, JsonObject.class);
        JsonObject jdata = json.get("data").getAsJsonObject();
        long fileSize = jdata.get("file_size").getAsLong();
        String fileType = jdata.get("file_mimetype").getAsString();

        //Authentication
        //TODO: Do we have to check for permission here, like in encrypt file??
        UserData authData;
        if (auth.length() < 20) {
            String password = Core.getStringParameter(request, "password", true);
            authData = new UserData();
            authData.encr_password = password;
            authData.userid = userId;
        } else {
            authData = new UserData(auth, cookie);
        }
        if (authData.userid == -1) {
            sendFail(response, GetText.gettext("Unable to decode file, bad authorization", lang));
            return;
        }

        //New JSON
        String description = (jdata.get("description") == null) ? "" : jdata.get("description").getAsString();
        JsonObject newJsonContent = new JsonObject();
        newJsonContent.add("folder_id", new JsonPrimitive(folderId));
        newJsonContent.addProperty("description", description);
        String newjson = newJsonContent.toString();
        JsonStreamHandler jsonHandler = new JsonStreamHandler();


        try(InputStream jsonStream = new ByteArrayInputStream(newjson.getBytes()))
        {
            //De-Crypting the item
            logger.debug(String.format("Decrypting file %s in folder %d ...",
                         fileId,
                         folderId));
            StopWatch sw = new StopWatch();
            sw.start();
            FileDecrypter fileDecrypter = new FileDecrypter(ap);

            JsonObject retJson = fileDecrypter.Decrypt(fileId,
                                                     fileSize,
                                                     folderId,
                                                     version,
                                                     fileName,
                                                     userId,
                                                     cid,
                                                     authData.encr_password,
                                                     encrExtra,
                                                     fileType,
                                                     jsonStream,
                                                     jsonHandler);
            sw.stop();
            long ms = sw.getTime();
            logger.debug(String.format("Decryption done for file %s in folder %d [Time: %d ms]",
                         fileId,
                         folderId,
                         ms));

            //deleting the old file
            ap.deleteItem(fileId, folderId);

          //sending response
          sendOK(response, retJson.toString());
        }
    }

    public void getShares(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) throws GuardMissingParameter {
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        String itemID = Core.getStringParameter(request, "id", true);
        try {
            Fileshare fs = new Fileshare();
            JsonArray json = fs.getShares(itemID, userid, cid);
            if (json == null) {
                com.openexchange.guard.util.Core.sendFail(response, "NOTFOUND");
            } else {
                Gson gson = new Gson();
                String json_string = gson.toJson(json);
                com.openexchange.guard.util.Core.sendOK(response, json_string);
            }
        } catch (Exception ex) {
            logger.error("Error getting shared for user " + userid + " in context " + cid + " and item " + itemID, ex);
        }
    }

    public void getfolder(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) {
        try {
            Api ap = new Api(oxCookie, request);
            String data = ap.getFolder(Integer.parseInt(request.getParameter("id")));
            if (!data.contains("error")) {
                com.openexchange.guard.util.Core.sendOK(response, data);
                return;
            }

        } catch (Exception ex) {
            logger.error("Error getting folder " + request.getParameter("id"), ex);
        }
        com.openexchange.guard.util.Core.sendFail(response, "error");
    }

    public void addFolderShare(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) {
        try {
            Api ap = new Api(oxCookie, request);
            int id = Integer.parseInt(request.getParameter("id"));
            String data = ap.getFolder(id);
            if (data.contains("error")) {
                com.openexchange.guard.util.Core.sendFail(response, "error retrieving folder perm");
                return;
            }
            String[] users = request.getParameter("users").split(",");
            JsonArray permissions = getPermissions(data);
            for (int i = 0; i < users.length; i++) {
                int user = Integer.parseInt(users[i]);
                boolean found = false;
                permloop: for (int j = 0; j < permissions.size(); j++) {
                    if (permissions.get(j).getAsJsonObject().get("entity").getAsInt() == user) {
                        // update
                        int bits = permissions.get(j).getAsJsonObject().get("bits").getAsInt();
                        bits = bits | 257;
                        JsonObject perm = permissions.get(j).getAsJsonObject();
                        perm.addProperty("bits", bits);
                        JsonArray newpermissions = new JsonArray();
                        for (int c = 0; c < permissions.size(); c++) {
                            if (c != j) {
                                newpermissions.add(permissions.get(c).getAsJsonObject());
                            }
                        }
                        newpermissions.add(perm);
                        permissions = newpermissions;
                        found = true;
                        break permloop;
                    }
                }
                if (!found) {
                    JsonObject newperm = new JsonObject();
                    newperm.addProperty("entity", user);
                    newperm.addProperty("bits", 257);
                    newperm.addProperty("group", false);
                    permissions.add(newperm);
                }
            }
            Gson gson = new Gson();
            JsonObject newdata = new JsonObject();
            newdata.add("permissions", permissions);
            String addresult = ap.updateFolder(id, gson.toJson(newdata));
            if (addresult.contains("error")) {
                logger.error("failed to add share");
                logger.error(addresult);
                com.openexchange.guard.util.Core.sendFail(response, "error adding folder perm");
                return;
            }
            com.openexchange.guard.util.Core.sendOK(response, "OK");
            return;
        } catch (Exception ex) {
            logger.error("Error adding folder share", ex);
        }
        com.openexchange.guard.util.Core.sendFail(response, "error");
    }

    private JsonArray getPermissions(String data) {
        Gson gson = new Gson();
        JsonObject json = gson.fromJson(data, JsonObject.class);
        JsonObject jdata = json.get("data").getAsJsonObject();
        JsonArray permissions = jdata.get("permissions").getAsJsonArray();
        return (permissions);
    }

    public void share(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) {
        try {
            JsonObject jdata = Core.getJSON(request);
            String auth = request.getParameter("auth");
            String password = jdata.get("password").getAsString();
            if (password.equals("undefined")) {
                password = null;
            }
            if (auth != null) {
                if (!auth.equals("null")) {
                    UserData data = new UserData(auth, oxCookie);
                    if (data.ox_password != null) {
                        password = data.ox_password;
                    }
                }
            }
            if (password == null) {
                com.openexchange.guard.util.Core.sendFail(response, "Fail");
                return;
            }
            int cid = Integer.parseInt(request.getParameter("cid"));
            int userid = Integer.parseInt(request.getParameter("userid"));
            String itemID = request.getParameter("id");
            String name = jdata.get("name").getAsString();
            String email = jdata.get("email").getAsString();
            Fileshare fs = new Fileshare();
            String sharedata = fs.share(itemID, name, email, userid, cid, password);
            if (sharedata == null) {
                com.openexchange.guard.util.Core.sendFail(response, "Bad Password");
                logger.info("Bad password for file share");
                return;
            }
            if (sharedata.equals("nsc")) {
                com.openexchange.guard.util.Core.sendFail(response, "Not same context");
                return;
            }
            com.openexchange.guard.util.Core.sendOK(response, sharedata);

        } catch (Exception ex) {
            logger.error("Error sharing", ex);
        }
    }

    public void removeShare(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) {
        String email = request.getParameter("email");
        String itemID = request.getParameter("id");
        try {
            Api ap = new Api(oxCookie, request);
            if (!ap.verifyLogin())
            {
                return; // Make sure legit log into UI
            }
            int cid = Integer.parseInt(request.getParameter("cid"));
            int userid = Integer.parseInt(request.getParameter("userid"));
            Fileshare fs = new Fileshare();
            if (fs.removeShare(itemID, userid, cid, email)) {
                com.openexchange.guard.util.Core.sendOK(response, "OK");
            } else {
                com.openexchange.guard.util.Core.sendFail(response, "Fail");
            }

        } catch (Exception ex) {
            logger.error("Error removing share for email " + email + " and item " + itemID, ex);
        }
    }

    public void addExtraPassword(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) throws GuardMissingParameter {

        JsonObject rdata = Core.getJSON(request);
        String auth = Core.getStringParameter(request, "auth");
        String password = Core.getStringFromJson(rdata, "password");
        if (password != null) {
            if (password.equals("undefined")) {
                password = null;
            }
        }

        if (auth != null) {
            UserData data = new UserData(auth, oxCookie);
            if (data.ox_password != null) {
                password = data.ox_password;
            }
        }
        if (password == null) {
            com.openexchange.guard.util.Core.sendFail(response, "Fail");
            return;
        }
        int userid = Core.getIntFromJson(rdata, "userid", true);
        int cid = Core.getIntFromJson(rdata, "cid", true);
        String id = Core.getStringFromJson(rdata, "id", true);
        int folder = Core.getIntFromJson(rdata, "folder", true);
        int version = Core.getIntFromJson(rdata, "version", true);
        String sessionID = Core.getStringParameter(request, "session", true);
        String extraPass = Core.getStringFromJson(rdata, "extrapass", true);
        if ((extraPass == null) || (extraPass.length() < 3)) {
            com.openexchange.guard.util.Core.sendFail(response, "Fail");
            return;
        }
        try {
            // Pull old encrypted file and it's json data
            Api ap = new Api(oxCookie, request);
            EncryptedObj file = ap.getInfoStore(id, folder, version);
            String data = ap.getInfoItemJson(id, folder, version);
            Gson gson = new Gson();
            JsonObject json = gson.fromJson(data, JsonObject.class);
            JsonObject jdata = json.get("data").getAsJsonObject();

            // Start creating new file
            Attachment encrfile = new Attachment();
            encrfile.filename = jdata.get("filename").getAsString().replace(".grd", ".grd2");
            encrfile.type = jdata.get("file_mimetype").getAsString();
            String description = (jdata.get("description") == null) ? "" : jdata.get("description").getAsString();
            // String newjson = "{\"folder_id\" : \"" + folder + "\",\"description\":\"" + description + "\"}";

            Crypto crypt = new Crypto();
            String decoded = crypt.decodeEncrObj(userid, cid, file, password); // Get base64 encoded decoded item
            if (decoded != null) {
                String itemID = EncrLib.getUUID();
                String newjson = getFileJson(itemID, encrfile.type, folder, description, userid, cid);
                encrfile.encrContent = crypt.encrItem(userid, cid, null, crypt.addExtraPass(decoded, extraPass, itemID), 0, 2, "", itemID);
                String params = "files?action=new&filename=" + URLEncoder.encode(encrfile.filename) + "&session=" + sessionID;
                ArrayList<Attachment> attachments = new ArrayList<Attachment>();
                attachments.add(encrfile);
                String resp = ap.filePost(newjson, params, attachments);
                if (!com.openexchange.guard.util.Core.checkFail(resp)) {
                    ap.deleteItem(id, folder);
                }
                com.openexchange.guard.util.Core.sendOK(response, "OK");
                return;
            }
            com.openexchange.guard.util.Core.sendFail(response, "fail");
        } catch (Exception ex) {
            logger.error("Error adding extra password", ex);
            Core.sendFail(response, "fail");
        }
    }
}
